package ru.yandex.solomon.dataproxy.client;

import java.net.ConnectException;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.net.HostAndPort;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.GrpcClientOptions;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.dataproxy.DataProxyServiceGrpc;
import ru.yandex.monitoring.dataproxy.FindRequest;
import ru.yandex.monitoring.dataproxy.FindResponse;
import ru.yandex.monitoring.dataproxy.LabelKeysRequest;
import ru.yandex.monitoring.dataproxy.LabelKeysResponse;
import ru.yandex.monitoring.dataproxy.LabelValuesRequest;
import ru.yandex.monitoring.dataproxy.LabelValuesResponse;
import ru.yandex.monitoring.dataproxy.MetricNamesRequest;
import ru.yandex.monitoring.dataproxy.MetricNamesResponse;
import ru.yandex.monitoring.dataproxy.ReadManyRequest;
import ru.yandex.monitoring.dataproxy.ReadManyResponse;
import ru.yandex.monitoring.dataproxy.ReadOneRequest;
import ru.yandex.monitoring.dataproxy.ReadOneResponse;
import ru.yandex.monitoring.dataproxy.ResolveManyRequest;
import ru.yandex.monitoring.dataproxy.ResolveManyResponse;
import ru.yandex.monitoring.dataproxy.ResolveOneRequest;
import ru.yandex.monitoring.dataproxy.ResolveOneResponse;
import ru.yandex.monitoring.dataproxy.UniqueLabelsRequest;
import ru.yandex.monitoring.dataproxy.UniqueLabelsResponse;
import ru.yandex.solomon.util.host.DnsResolver;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class GrpcDataProxyClient implements DataProxyClient {

    private static final Logger logger = LoggerFactory.getLogger(GrpcDataProxyClient.class);

    private final NearestList<GrpcTransport> nearestList;
    private final ScheduledExecutorService scheduler;

    private final ActorWithFutureRunner refreshListActor;
    private volatile ScheduledFuture<?> refreshFuture;
    private volatile boolean closed = false;

    private volatile GrpcTransport currentTransport;

    @VisibleForTesting
    public GrpcDataProxyClient(HostAndPort address, GrpcClientOptions opts, DnsResolver resolver) {
        this(List.of(address), opts, resolver);
    }

    public GrpcDataProxyClient(List<HostAndPort> addresses, GrpcClientOptions opts, DnsResolver resolver) {
        this(addresses, opts, new NearestList<>(HostUtils.getFqdn(), resolver));
    }

    private GrpcDataProxyClient(List<HostAndPort> addresses, GrpcClientOptions opts, NearestList<GrpcTransport> nearestList) {
        Preconditions.checkState(!addresses.isEmpty(), "DataProxy client invalid configuration, addresses list is empty");

        this.nearestList = nearestList;

        for (var address : addresses) {
            nearestList.add(new GrpcTransport(address, opts));
        }

        this.scheduler = opts.getTimer().orElseThrow(() -> new IllegalArgumentException("Timer is not configured for dataproxy client"));
        var executor = opts.getRpcExecutor().orElseThrow(() -> new IllegalArgumentException("RpcExecutor is not configured for dataproxy client"));
        this.refreshListActor = new ActorWithFutureRunner(this::refreshList, executor);

        refreshListActor.schedule();
        nearestList.initializedFuture().join();

        var nearestSnapshot = nearestList.snapshot();

        this.currentTransport = nearestSnapshot.getNearest();

        logger.info("connect to DataProxy at {} with fallback to {}",
                currentTransport.getAddress(),
                nearestSnapshot.getAllFallbacks());
    }

    private CompletableFuture<Void> refreshList() {
        if (closed) {
            return CompletableFuture.completedFuture(null);
        }
        return CompletableFutures.safeCall(nearestList::rearrange)
                .whenComplete(this::reschedule);
    }

    private void reschedule(Void unused, Throwable throwable) {
        if (closed) {
            return;
        }
        refreshFuture = scheduler.schedule(refreshListActor::schedule, 1, TimeUnit.MINUTES);
    }

    private void trySwitchBackToLocal(NearestList.Snapshot<GrpcTransport> snapshot) {
        // intentionally check availability by isConnected(), to make CircuitBreaker to close again
        var local = snapshot.getNearest();
        if (local != null && currentTransport != local && local.isConnected()) {
            logger.info("switch back to local DataProxy at {}", local.getAddress());
            currentTransport = local;
        }
    }

    private boolean switchToAnyAvailableHost(NearestList.Snapshot<GrpcTransport> snapshot) {
        // prefer local DataProxy
        var local = snapshot.getNearest();
        if (local != null && local.isReady()) {
            logger.info("switch to local DataProxy at {}", local.getAddress());
            currentTransport = local;
            return true;
        }

        var fallback = snapshot.getFallbackCandidates();

        if (fallback.size() != 0) {
            GrpcTransport chosen = fallback.get(ThreadLocalRandom.current().nextInt(fallback.size()));
            logger.info("switch to non local DataProxy at {}", chosen.getAddress());
            currentTransport = chosen;
            return true;
        }

        logger.warn("all DataProxy hosts are unavailable");
        return false;
    }

    @Override
    public CompletableFuture<FindResponse> find(FindRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getFindMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<ResolveOneResponse> resolveOne(ResolveOneRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getResolveOneMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<ResolveManyResponse> resolveMany(ResolveManyRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getResolveManyMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<MetricNamesResponse> metricNames(MetricNamesRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getMetricNamesMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<LabelKeysResponse> labelKeys(LabelKeysRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getLabelKeysMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<LabelValuesResponse> labelValues(LabelValuesRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getLabelValuesMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<UniqueLabelsResponse> uniqueLabels(UniqueLabelsRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getUniqueLabelsMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<ReadOneResponse> readOne(ReadOneRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getReadOneMethod(), request, deadlineMillis);
    }

    @Override
    public CompletableFuture<ReadManyResponse> readMany(ReadManyRequest request, long deadlineMillis) {
        return callWithRetry(DataProxyServiceGrpc.getReadManyMethod(), request, deadlineMillis);
    }

    private static boolean needToSwitchHost(Throwable t) {
        var cause = CompletableFutures.unwrapCompletionException(t);
        if (cause instanceof ConnectException) {
            return true;
        }

        if (cause instanceof StatusRuntimeException) {
            var code = ((StatusRuntimeException) cause).getStatus().getCode();
            return code == Status.Code.UNAVAILABLE;
        }

        return false;
    }

    private <ReqT, RespT> CompletableFuture<RespT> callWithRetry(
            MethodDescriptor<ReqT, RespT> method,
            ReqT request,
            long deadlineMillis)
    {
        var promise = new CompletableFuture<RespT>();
        currentTransport.unaryCall(method, request, deadlineMillis)
                .whenComplete((response, throwable) -> {
                    var snapshot = nearestList.snapshot();

                    if (throwable == null) {
                        trySwitchBackToLocal(snapshot);
                        promise.complete(response);
                        return;
                    }

                    if (needToSwitchHost(throwable) && switchToAnyAvailableHost(snapshot)) {
                        // one-shot retry if tha last used node is unavailable and
                        // there is available fallback node
                        var future = currentTransport.unaryCall(method, request, deadlineMillis);
                        CompletableFutures.whenComplete(future, promise);
                    } else {
                        promise.completeExceptionally(throwable);
                    }
                });
        return promise;
    }

    @Override
    public void close() {
        closed = true;
        var refreshFuture = this.refreshFuture;
        if (refreshFuture != null) {
            refreshFuture.cancel(true);
        }
        nearestList.snapshot().getAllTransports().forEach(GrpcTransport::close);
    }
}
