package ru.yandex.solomon.dataproxy.client;

import java.util.ArrayList;
import java.util.List;

import ru.yandex.monitoring.dataproxy.FindResponse;
import ru.yandex.monitoring.dataproxy.Metric;
import ru.yandex.monitoring.dataproxy.MetricKeyPooled;
import ru.yandex.monitoring.dataproxy.MetricPooled;
import ru.yandex.monitoring.dataproxy.ResolveManyResponse;
import ru.yandex.monitoring.dataproxy.ResolveOneResponse;
import ru.yandex.monitoring.dataproxy.StockpileId;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.MetricKeyPartial;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.util.protobuf.StringPool;

/**
 * @author Sergey Polovko
 */
public final class MetricConverter {
    private MetricConverter() {}

    public static List<MetricKey> toMetricKeys(FindResponse resp) {
        return metricKeys(
                StringPool.fromProto(resp.getStringPool()),
                resp.getReplicaName1(),
                resp.getReplicaName2(),
                resp.getMetricsList());
    }

    public static List<MetricKey> toMetricKeys(ResolveManyResponse resp) {
        return metricKeys(
                StringPool.fromProto(resp.getStringPool()),
                resp.getReplicaName1(), resp.getReplicaName2(),
                resp.getMetricsList());
    }

    public static MetricKey toMetricKey(ResolveOneResponse resp) {
        Metric metric = resp.getMetric();
        var labelsBuilder = Labels.builder(metric.getLabelsCount() / 2);
        return metricKey(labelsBuilder, resp.getReplicaName1(), resp.getReplicaName2(), metric);
    }

    public static MetricKey toMetricKey(StringPool strings, MetricPooled metric) {
        var labelsBuilder = Labels.builder(metric.getLabelsIdxCount() / 2);

        MetricType type = MetricTypeConverter.fromProto(metric.getType());
        String name = strings.get(metric.getNameIdx());
        Labels labels = LabelsConverter.fromIntArray(strings, labelsBuilder, metric.getLabelsIdxList());

        return new MetricKey(type, name, labels, 0, List.of());
    }

    public static List<MetricKeyPooled> fromMetrics(StringPool.Builder builder, List<MetricKeyPartial> metrics) {
        var keys = new ArrayList<MetricKeyPooled>(metrics.size());
        for (MetricKeyPartial metric : metrics) {
            keys.add(MetricKeyPooled.newBuilder()
                    .setNameIdx(builder.put(metric.getName()))
                    .addAllLabelsIdx(LabelsConverter.toIntArray(builder, metric.getLabels()))
                    .build());
        }
        return keys;
    }

    public static List<MetricKeyPooled> fromLabelsList(StringPool.Builder builder, List<Labels> labelsList) {
        int emptyIdx = builder.put("");
        var keys = new ArrayList<MetricKeyPooled>(labelsList.size());
        for (Labels labels : labelsList) {
            keys.add(MetricKeyPooled.newBuilder()
                    .setNameIdx(emptyIdx)
                    .addAllLabelsIdx(LabelsConverter.toIntArray(builder, labels))
                    .build());
        }
        return keys;
    }

    private static List<MetricKey> metricKeys(
            StringPool strings,
            String replicaName1, String replicaName2,
            List<MetricPooled> protoMetrics)
    {
        var labelsBuilder = Labels.builder(Labels.MAX_LABELS_COUNT);
        var metrics = new ArrayList<MetricKey>(protoMetrics.size());
        for (MetricPooled protoMetric : protoMetrics) {
            metrics.add(metricKey(strings, labelsBuilder, replicaName1, replicaName2, protoMetric));
        }
        return metrics;
    }

    static MetricKey metricKey(
            StringPool strings, LabelsBuilder labelsBuilder,
            String replicaName1, String replicaName2,
            MetricPooled metric)
    {
        String name = strings.get(metric.getNameIdx());
        Labels labels = LabelsConverter.fromIntArray(strings, labelsBuilder, metric.getLabelsIdxList());
        MetricType type = MetricTypeConverter.fromProto(metric.getType());

        var stockpileKeys = stockpileKeys(
                replicaName1, replicaName2,
                metric.getStockpileReplica1(), metric.getStockpileReplica2());

        return new MetricKey(type, name, labels, 0, stockpileKeys);
    }

    static MetricKey metricKey(LabelsBuilder labelsBuilder, String replicaName1, String replicaName2, Metric metric) {
        String name = metric.getName();
        Labels labels = LabelsConverter.fromStringArray(labelsBuilder, metric.getLabelsList());
        MetricType type = MetricTypeConverter.fromProto(metric.getType());

        var stockpileKeys = stockpileKeys(
                replicaName1, replicaName2,
                metric.getStockpileReplica1(), metric.getStockpileReplica2());

        return new MetricKey(type, name, labels, 0, stockpileKeys);
    }

    static List<StockpileKey> stockpileKeys(String name1, String name2, StockpileId id1, StockpileId id2) {
        // avoid useless allocations

        if (id1.getLocalId() != 0) {
            StockpileKey k1 = stockpileKey(name1, id1);
            return (id2.getLocalId() != 0)
                    ? List.of(k1, stockpileKey(name2, id2))
                    : List.of(k1);
        }

        if (id2.getLocalId() != 0) {
            return List.of(stockpileKey(name2, id2));
        }

        return List.of();
    }

    private static StockpileKey stockpileKey(String replicaName, StockpileId id) {
        return new StockpileKey(replicaName, id.getShardId(), id.getLocalId());
    }
}
