package ru.yandex.solomon.dataproxy.client;

import java.net.Inet6Address;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.grpc.utils.Transport;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.host.DnsResolver;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class NearestList<T extends Transport> {
    private final String localFqdn;
    private final DnsResolver resolver;
    private final CompletableFuture<Void> initialized = new CompletableFuture<>();
    private final int shuffleKey;

    private static final int INET6_ADDRESS_BYTES = 16;

    static record TransportWithPriority<T extends Transport>(T transport, int priority) {
    }

    private volatile List<TransportWithPriority<T>> transportsWithPriorities = new ArrayList<>();

    public NearestList(String localFdqn, DnsResolver resolver) {
        this.localFqdn = localFdqn;
        this.resolver = resolver;
        this.shuffleKey = localFdqn.hashCode();
    }

    public void add(T transport) {
        transportsWithPriorities.add(new TransportWithPriority<>(transport, 0));
    }

    public Snapshot<T> snapshot() {
        return new Snapshot<>(transportsWithPriorities);
    }

    public CompletableFuture<Void> initializedFuture() {
        return initialized;
    }

    public CompletableFuture<Void> rearrange() {
        try {
            List<T> transports = transportsWithPriorities.stream()
                    .map(TransportWithPriority::transport)
                    .collect(Collectors.toList());

            List<CompletableFuture<Inet6Address>> futures = new ArrayList<>(transports.size() + 1);
            futures.add(resolver.resolve(localFqdn));
            for (var transport : transports) {
                futures.add(resolver.resolve(transport.getAddress().getHost()));
            }
            return CompletableFutures.allOf(futures)
                    .thenAccept(addresses -> doRearrange(transports, addresses))
                    .thenAccept(aVoid -> initialized.complete(null));
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private void doRearrange(List<T> transports, ListF</* @Nullable */ Inet6Address> addresses) {
        @Nullable Inet6Address local = addresses.get(0);
        List</* @Nullable */ Inet6Address> others = addresses.subList(1, addresses.size());

        List<TransportWithPriority<T>> transportsWithPriorities = new ArrayList<>(transports.size());
        for (int i = 0; i < transports.size(); i++) {
            transportsWithPriorities.add(new TransportWithPriority<>(
                    transports.get(i),
                    samePrefixLenBytes(local, others.get(i))
            ));
        }

        // If there are several nearest transports, shuffle them stably,
        transportsWithPriorities.sort(Comparator.<TransportWithPriority<T>>
                comparingInt(TransportWithPriority::priority).reversed()
                .thenComparing(tr -> tr.transport().getAddress().getHost().hashCode() ^ shuffleKey));

        this.transportsWithPriorities = transportsWithPriorities;
    }

    private int samePrefixLenBytes(@Nullable Inet6Address local, @Nullable Inet6Address other) {
        if (local == null || other == null) {
            return 0;
        }
        if (local.equals(other) || other.isLoopbackAddress()) {
            return INET6_ADDRESS_BYTES;
        }

        byte[] localBytes = local.getAddress();
        byte[] otherBytes = other.getAddress();

        if (localBytes.length != otherBytes.length) {
            return 0;
        }

        for (int i = 0; i < localBytes.length; i++) {
            if (localBytes[i] != otherBytes[i]) {
                return i;
            }
        }
        return localBytes.length;
    }

    public static class Snapshot<T extends Transport> {
        private final List<TransportWithPriority<T>> transports;

        public Snapshot(List<TransportWithPriority<T>> transports) {
            this.transports = transports;
        }

        @Nullable
        public T getNearest() {
            if (transports.isEmpty()) {
                return null;
            }

            return transports.get(0).transport();
        }

        public List<T> getAllTransports() {
            return transports.stream()
                    .map(TransportWithPriority::transport)
                    .collect(Collectors.toList());
        }

        public List<T> getAllFallbacks() {
            return transports.stream()
                    .skip(1)
                    .map(TransportWithPriority::transport)
                    .collect(Collectors.toList());
        }

        public List<T> getFallbackCandidates() {
            var ready = transports.stream()
                    .skip(1)
                    .filter(twp -> twp.transport().isReady())
                    .collect(Collectors.toList());
            if (ready.isEmpty()) {
                return List.of();
            }

            int topPriority = ready.get(0).priority();

            return ready.stream()
                    .filter(twp -> twp.priority() == topPriority)
                    .map(TransportWithPriority::transport)
                    .collect(Collectors.toList());
        }
    }
}
