package ru.yandex.solomon.dataproxy.client;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.channel.EventLoopGroup;
import io.netty.resolver.dns.DnsNameResolver;
import io.netty.resolver.dns.DnsNameResolverBuilder;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.util.NettyUtils;
import ru.yandex.solomon.util.host.DnsResolver;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class NettyDnsResolver implements DnsResolver {
    private static final Logger logger = LoggerFactory.getLogger(NettyDnsResolver.class);

    private final DnsNameResolver resolver;

    public NettyDnsResolver(EventLoopGroup eventLoopGroup) {
        this.resolver = new DnsNameResolverBuilder(eventLoopGroup.next())
                .channelType(NettyUtils.clientDatagramChannelForEventLoop(eventLoopGroup))
                .build();
    }

    @Override
    public CompletableFuture<Inet6Address> resolve(String fqdn) {
        CompletableFuture<Inet6Address> future = new CompletableFuture<>();
        resolver.resolveAll(fqdn)
                .addListener((GenericFutureListener<Future<List<InetAddress>>>) result -> {
                    if (result.isSuccess()) {
                        var addrV6 = (Inet6Address) result.getNow().stream()
                                .filter(addr -> addr instanceof Inet6Address)
                                .findFirst()
                                .orElseGet(() -> {
                                    logger.error("Failed to resolve ipv6 address for: " + fqdn);
                                    return null;
                                });
                        future.complete(addrV6);
                    } else {
                        logger.error("Failed to resolve address: " + fqdn, result.cause());
                        future.complete(null);
                    }
                });

        return future;
    }
}
