package ru.yandex.solomon.dataproxy.client;

import java.util.List;
import java.util.concurrent.TimeUnit;

import com.google.common.net.HostAndPort;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.monitoring.dataproxy.FindRequest;
import ru.yandex.monitoring.dataproxy.FindResponse;
import ru.yandex.monitoring.dataproxy.MetricKeyPooled;
import ru.yandex.monitoring.dataproxy.MetricPooled;
import ru.yandex.monitoring.dataproxy.ReadManyRequest;
import ru.yandex.monitoring.dataproxy.ReadManyResponse;
import ru.yandex.monitoring.dataproxy.ReadOneRequest;
import ru.yandex.monitoring.dataproxy.ReadOneResponse;
import ru.yandex.monitoring.dataproxy.ResolveOneRequest;
import ru.yandex.monitoring.dataproxy.ResolveOneResponse;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.common.StringPool.Compression;
import ru.yandex.solomon.config.thread.StubThreadPoolProvider;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationAggregationSummary;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.ConcatAggrGraphDataIterable;
import ru.yandex.solomon.util.protobuf.StringPool;

import static java.util.stream.Collectors.toList;

/**
 * @author Sergey Polovko
 */
@YaIgnore
public class DataProxyClientTest {

    private StubThreadPoolProvider threadPoolProvider;
    private DataProxyClient client;

    @Before
    public void setUp() throws Exception {
        threadPoolProvider = new StubThreadPoolProvider();
        var clientOptions = DefaultClientOptions.newBuilder()
                .setTimer(threadPoolProvider.getSchedulerExecutorService())
                .setRpcExecutor(threadPoolProvider.getIOExecutor())
                .build();
        client = new GrpcDataProxyClient(
                HostAndPort.fromParts("localhost", 5770),
                clientOptions,
                new NettyDnsResolver(threadPoolProvider.getIOExecutor()));
    }

    @After
    public void tearDown() throws Exception {
        client.close();
        threadPoolProvider.close();
    }

    @Test
    public void find() {
        long now = System.currentTimeMillis();

        FindRequest request = FindRequest.newBuilder()
                .setProjectId("solomon")
                .setSelectors("{" +
                        "cluster=\"production\", " +
                        "service=\"coremon\", " +
                        "host=\"solomon-fetcher-sas-001\", " +
                        "sensor=\"jvm.threadPool.activeThreads\"" +
                        "}")
                .setFromMillis(now - TimeUnit.MINUTES.toMillis(15))
                .setToMillis(now)
                .setLimit(20)
                .build();

        FindResponse response = client.find(request, 0).join();
        System.out.println("-------------------");
        System.out.println(response);
        System.out.println("-------------------");

        List<MetricKey> metricKeys = MetricConverter.toMetricKeys(response);
        for (MetricKey key : metricKeys) {
            System.out.println(key);
        }
    }

    @Test
    public void resolveOne() {
        ResolveOneRequest request = ResolveOneRequest.newBuilder()
                .setProjectId("solomon")
                .setName("")
                .addAllLabels(List.of(
                        "project", "solomon",
                        "cluster", "production",
                        "service", "coremon",
                        "sensor", "proc.self.rssBytes",
                        "host", "Sas"
                ))
                .build();

        ResolveOneResponse response = client.resolveOne(request, 0).join();

        System.out.println("-------------------");
        System.out.println(response);
        System.out.println("-------------------");

        MetricKey metricKey = MetricConverter.toMetricKey(response);
        System.out.println(metricKey);
    }

    @Test
    public void readOne() {
        long now = System.currentTimeMillis();
        now -= now % 15_000;

        ReadOneRequest request = ReadOneRequest.newBuilder()
                .setProjectId("solomon")
                .setName("")
                .addAllLabels(List.of(
                        "project", "solomon",
                        "cluster", "production",
                        "service", "coremon",
                        "sensor", "proc.self.rssBytes",
                        "host", "Sas"
                ))
                .setFromMillis(now - TimeUnit.MINUTES.toMillis(5))
                .setToMillis(now)
                .setMaxTimeSeriesFormat(StockpileFormat.CURRENT.getFormat())
                .build();

        ReadOneResponse response = client.readOne(request, 0).join();

        System.out.println("-------------------");
        System.out.println(response);
        System.out.println("-------------------");

        TimeSeries timeseries = response.getTimeSeries();
        StockpileFormat format = StockpileFormat.byNumber(timeseries.getFormatVersion());
        AggrGraphDataIterable graphData = decode(format, timeseries);
        System.out.println("records: " + graphData.getRecordCount());

        AggrGraphDataListIterator it = graphData.iterator();
        for (AggrPoint point = new AggrPoint(); it.next(point); ) {
            System.out.println(point);
        }
    }

    @Test
    public void readMany() {
        long now = System.currentTimeMillis();
        now -= now % 15_000;

        ReadManyResponse response;
        {
            var strings = StringPool.newBuilder();

            List<Integer> commonLabels = List.of(
                    strings.put("cluster"),
                    strings.put("production"),
                    strings.put("service"),
                    strings.put("coremon"));

            List<MetricKeyPooled> metricKeys = MetricConverter.fromLabelsList(strings, List.of(
                    Labels.of("host", "Sas", "sensor", "proc.self.rssBytes"),
                    Labels.of("host", "Vla", "sensor", "proc.self.rssBytes")
            ));

            ReadManyRequest request = ReadManyRequest.newBuilder()
                    .setProjectId("solomon")
                    .setResolvedKeys(ReadManyRequest.ResolvedKeys.newBuilder()
                            .setStringPool(strings.buildProto(Compression.LZ4))
                            .addAllCommonLabelsIdx(commonLabels)
                            .addAllMetricKeys(metricKeys)
                            .build())
                    .setFromMillis(now - TimeUnit.MINUTES.toMillis(5))
                    .setToMillis(now)
                    .setMaxTimeSeriesFormat(StockpileFormat.CURRENT.getFormat())
                    .addOperations(Operation.newBuilder()
                            .setSummary(OperationAggregationSummary.newBuilder()
                                    .addAggregations(Aggregation.AVG)
                                    .addAggregations(Aggregation.COUNT)
                                    .addAggregations(Aggregation.SUM)
                                    .build())
                            .build())
                    .addOperations(Operation.newBuilder()
                            .setDropTimeseries(OperationDropTimeSeries.getDefaultInstance())
                            .build())
                    .build();

            response = client.readMany(request, 0).join();
        }

        System.out.println("-------------------");
        System.out.println(response);
        System.out.println("-------------------");

        var strings = StringPool.fromProto(response.getStringPool());
        var sb = new StringBuilder();
        for (ReadManyResponse.MetricData metricData : response.getMetricsList()) {
            MetricPooled metric = metricData.getMetric();
            sb.append(metric.getType()).append(' ');
            sb.append(strings.get(metric.getNameIdx())).append('{');
            for (int i = 0; i < metric.getLabelsIdxCount(); ) {
                sb.append(strings.get(metric.getLabelsIdx(i++)));
                sb.append('=');
                sb.append(strings.get(metric.getLabelsIdx(i++)));
            }
            sb.append("} [");

            TimeSeries timeseries = metricData.getTimeSeries();
            StockpileFormat format = StockpileFormat.byNumber(timeseries.getFormatVersion());
            AggrGraphDataListIterator it = decode(format, timeseries).iterator();
            for (AggrPoint point = new AggrPoint(); it.next(point); ) {
                sb.append(point).append(", ");
            }
            sb.append(']');

            System.out.println(sb.toString());
            sb.setLength(0);
        }
    }

    private static AggrGraphDataIterable decode(StockpileFormat format, TimeSeries timeseries) {
        var serializer = MetricArchiveNakedSerializer.serializerForFormatSealed(format);
        var graphDataList = timeseries.getChunksList().stream()
                .map(chunk -> {
                    if (chunk.getPointCount() == 0) {
                        return (AggrGraphDataIterable) MetricArchiveImmutable.empty;
                    }

                    return serializer.deserializeToEof(new StockpileDeserializer(chunk.getContent()));
                })
                .collect(toList());
        return ConcatAggrGraphDataIterable.of(graphDataList);
    }
}
