package ru.yandex.solomon.dataproxy.client;

import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.util.protobuf.StringPool;

/**
 * @author Sergey Polovko
 */
public class LabelsConverterTest {

    @Test
    public void toStringArray() {
        // empty
        {
            List<String> strings = LabelsConverter.toStringArray(Labels.empty());
            Assert.assertTrue(strings.isEmpty());
        }

        // single
        {
            List<String> strings = LabelsConverter.toStringArray(Labels.of("project", "solomon"));
            Assert.assertEquals(List.of("project", "solomon"), strings);
        }

        // multi (ordered)
        {
            List<String> strings = LabelsConverter.toStringArray(Labels.of("project", "solomon", "name", "cpuUsage"));
            Assert.assertEquals(List.of("name", "cpuUsage", "project", "solomon"), strings);
        }
    }

    @Test
    public void fromStringArray() {
        // empty
        {
            Labels labels = LabelsConverter.fromStringArray(Labels.builder(), List.of());
            Assert.assertEquals(Labels.empty(), labels);
        }

        // single
        {
            Labels labels = LabelsConverter.fromStringArray(Labels.builder(), List.of("project", "solomon"));
            Assert.assertEquals(Labels.of("project", "solomon"), labels);
        }

        // multi
        {
            Labels labels = LabelsConverter.fromStringArray(
                    Labels.builder(),
                    List.of("project", "solomon", "name", "cpuUsage"));
            Assert.assertEquals(Labels.of("project", "solomon", "name", "cpuUsage"), labels);
        }

        // odd items
        try {
            LabelsConverter.fromStringArray(Labels.builder(), List.of("project", "solomon", "name"));
            Assert.fail("expected exception was not thrown");
        } catch (ArrayIndexOutOfBoundsException e) {
            Assert.assertEquals("Index 3 out of bounds for length 3", e.getMessage());
        }
    }

    @Test
    public void toIntArray() {
        // empty
        {
            var builder = StringPool.newBuilder();
            List<Integer> ids = LabelsConverter.toIntArray(builder, Labels.empty());
            Assert.assertTrue(ids.isEmpty());
            Assert.assertEquals(0, builder.size());
        }

        // single
        {
            var builder = StringPool.newBuilder();
            List<Integer> ids = LabelsConverter.toIntArray(builder, Labels.of("project", "solomon"));
            Assert.assertEquals(List.of(1, 2), ids);
            Assert.assertEquals(2, builder.size());
        }

        // multi
        {
            var builder = StringPool.newBuilder();
            List<Integer> ids = LabelsConverter.toIntArray(builder, Labels.of("project", "solomon", "name", "cpuUsage"));
            Assert.assertEquals(List.of(1, 2, 3, 4), ids);
            Assert.assertEquals(4, builder.size());
        }
    }

    @Test
    public void fromIntArray() {
        StringPool strings = new StringPool("project", "cluster", "service", "solomon", "cpuUsage", "name");

        // empty
        {
            Labels labels = LabelsConverter.fromIntArray(strings, Labels.builder(), List.of());
            Assert.assertEquals(Labels.empty(), labels);
        }

        // single
        {
            Labels labels = LabelsConverter.fromIntArray(strings, Labels.builder(), List.of(0, 3));
            Assert.assertEquals(Labels.of("project", "solomon"), labels);
        }

        // multi
        {
            Labels labels = LabelsConverter.fromIntArray(strings, Labels.builder(), List.of(0, 3, 5, 4));
            Assert.assertEquals(Labels.of("project", "solomon", "name", "cpuUsage"), labels);
        }

        // odd items
        try {
            LabelsConverter.fromIntArray(strings, Labels.builder(), List.of(0, 1, 2));
            Assert.fail("expected exception was not thrown");
        } catch (ArrayIndexOutOfBoundsException e) {
            Assert.assertEquals("Index 3 out of bounds for length 3", e.getMessage());
        }
    }
}
