package ru.yandex.solomon.dataproxy.client;

import java.util.List;

import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import ru.yandex.monitoring.dataproxy.Metric;
import ru.yandex.monitoring.dataproxy.MetricPooled;
import ru.yandex.monitoring.dataproxy.StockpileId;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.model.StockpileKey;
import ru.yandex.solomon.util.protobuf.StringPool;

/**
 * @author Sergey Polovko
 */
public class MetricConverterTest {

    private static StringPool STRINGS;

    @BeforeClass
    public static void beforeClass() {
        STRINGS = new StringPool(
                "", "cluster", "production", "project", "solomon", "service", "coremon", "host",
                "solomon-fetcher-sas-001", "sensor", "jvm.threadPool.activeThreads", "threadPool",
                "Http", "CpuHighPriority", "MetabasePool", "ParsingThreadPool", "Scheduler");
    }

    @Test
    public void convertMetricPooled() {
        var metricProto = MetricPooled.newBuilder()
                .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                .setNameIdx(10)
                .addAllLabelsIdx(List.of(3, 4, 1, 2, 5, 6, 11, 12))
                .setStockpileReplica1(stockpileId(1, 2))
                .setStockpileReplica2(stockpileId(3, 4))
                .build();

        var metric = MetricConverter.metricKey(STRINGS, Labels.builder(), "sas", "vla", metricProto);
        Assert.assertEquals(MetricType.DGAUGE, metric.getType());
        Assert.assertEquals("jvm.threadPool.activeThreads", metric.getName());
        Assert.assertEquals(Labels.of(
                "project", "solomon",
                "cluster", "production",
                "service", "coremon",
                "threadPool", "Http"
        ), metric.getLabels());

        StockpileKey[] expectedStockpileKeys = {
                new StockpileKey("sas", 1, 2),
                new StockpileKey("vla", 3, 4),
        };
        Assert.assertArrayEquals(expectedStockpileKeys, metric.getStockpileKeys().toArray());
    }

    @Test
    public void convertMetric() {
        var metricProto = Metric.newBuilder()
                .setType(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE)
                .setName("jvm.threadPool.activeThreads")
                .addAllLabels(List.of(
                        "project", "solomon",
                        "cluster", "production",
                        "service", "coremon",
                        "threadPool", "Http"
                ))
                .setStockpileReplica1(stockpileId(1, 2))
                .setStockpileReplica2(stockpileId(3, 4))
                .build();

        var metric = MetricConverter.metricKey(Labels.builder(), "sas", "vla", metricProto);
        Assert.assertEquals(MetricType.DGAUGE, metric.getType());
        Assert.assertEquals("jvm.threadPool.activeThreads", metric.getName());
        Assert.assertEquals(Labels.of(
                "project", "solomon",
                "cluster", "production",
                "service", "coremon",
                "threadPool", "Http"
        ), metric.getLabels());

        StockpileKey[] expectedStockpileKeys = {
                new StockpileKey("sas", 1, 2),
                new StockpileKey("vla", 3, 4),
        };
        Assert.assertArrayEquals(expectedStockpileKeys, metric.getStockpileKeys().toArray());
    }

    @Test
    public void stockpileKeys() {
        StockpileId emptyId = StockpileId.getDefaultInstance();

        // all ids are empty
        {
            var keys = MetricConverter.stockpileKeys("sas", "vla", emptyId, emptyId);
            Assert.assertTrue(keys.isEmpty());
        }

        // second id is empty
        {
            var keys = MetricConverter.stockpileKeys("sas", "vla", stockpileId(1, 2), emptyId);
            Assert.assertEquals(1, keys.size());
            Assert.assertEquals(new StockpileKey("sas", 1, 2), keys.get(0));
        }

        // first id is empty
        {
            var keys = MetricConverter.stockpileKeys("sas", "vla", emptyId, stockpileId(1, 2));
            Assert.assertEquals(1, keys.size());
            Assert.assertEquals(new StockpileKey("vla", 1, 2), keys.get(0));
        }

        // all ids are non empty
        {
            var keys = MetricConverter.stockpileKeys("sas", "vla", stockpileId(1, 2), stockpileId(3, 4));
            Assert.assertEquals(2, keys.size());
            Assert.assertEquals(new StockpileKey("sas", 1, 2), keys.get(0));
            Assert.assertEquals(new StockpileKey("vla", 3, 4), keys.get(1));
        }
    }

    private static StockpileId stockpileId(int shardId, long localId) {
        return StockpileId.newBuilder().setShardId(shardId).setLocalId(localId).build();
    }
}
