package ru.yandex.solomon.dataproxy.client;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.HostAndPort;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.grpc.utils.Transport;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.host.DefaultDnsResolver;
import ru.yandex.solomon.util.host.DnsResolver;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class NearestListTest {

    private static final List<String> DATAPROXY_HOSTS = List.of(
            "gateway-myt-00.mon.yandex.net",
            "gateway-vla-00.mon.yandex.net",
            "gateway-sas-01.mon.yandex.net",
            "gateway-vla-02.mon.yandex.net",
            "gateway-myt-02.mon.yandex.net",
            "gateway-vla-01.mon.yandex.net",
            "gateway-sas-02.mon.yandex.net",
            "gateway-myt-01.mon.yandex.net",
            "gateway-sas-00.mon.yandex.net"
    );

    private static final List<String> ALERTING_HOSTS = List.of(
            "alerting-myt-00.mon.yandex.net",
            "alerting-vla-00.mon.yandex.net",
            "alerting-sas-01.mon.yandex.net",
            "alerting-vla-02.mon.yandex.net",
            "alerting-myt-02.mon.yandex.net",
            "alerting-vla-01.mon.yandex.net",
            "alerting-sas-02.mon.yandex.net",
            "alerting-myt-01.mon.yandex.net",
            "alerting-sas-00.mon.yandex.net"
    );

    private NearestList<FakeTransport> nearestList;
    private String selfFqdn;
    private PredefinedResolver resolver;

    private static class FakeTransport implements Transport {
        private final HostAndPort address;
        private boolean ready;
        private boolean connected;

        public FakeTransport(HostAndPort address) {
            this.address = address;
            this.ready = true;
            this.connected = true;
        }

        @Override
        public boolean isReady() {
            return ready;
        }

        @Override
        public boolean isConnected() {
            return connected;
        }

        @Override
        public HostAndPort getAddress() {
            return address;
        }

        public void setReady(boolean ready) {
            this.ready = ready;
        }

        public void setConnected(boolean connected) {
            this.connected = connected;
        }
    }

    private static class PredefinedResolver implements DnsResolver {
        private final HashMap<String, Inet6Address> predefined = new HashMap<>();

        @Override
        public CompletableFuture</* @Nullable */ Inet6Address> resolve(String fqdn) {
            return CompletableFutures.supplyAsync(() -> predefined.get(fqdn), ForkJoinPool.commonPool());
        }

        public void put(String key, @Nullable Inet6Address value) {
            predefined.put(key, value);
        }
    }

    @Before
    public void setUp() {
        this.resolver = new PredefinedResolver();

        this.resolver = new PredefinedResolver();
        for (var entry : STATIC_DNS_CACHE.entrySet()) {
            resolver.put(entry.getKey(), entry.getValue());
        }

        this.selfFqdn = "alerting-sas-01.mon.yandex.net";

        resolver.put(selfFqdn, resolver.resolve(selfFqdn).join());

        this.nearestList = new NearestList<>(selfFqdn, resolver);
        for (var host : DATAPROXY_HOSTS) {
            nearestList.add(new FakeTransport(HostAndPort.fromParts(host, 5770)));
        }
    }

    @After
    public void tearDown() {
    }

    @Test
    public void localFromSameHost() {
        nearestList.rearrange().join();
        var snapshot = nearestList.snapshot();
        Assert.assertEquals(selfFqdn.replace("alerting", "gateway"), snapshot.getNearest().getAddress().getHost());
    }

    @Test
    public void fallBackCandidatesFromSameDc() {
        nearestList.rearrange().join();
        var snapshot = nearestList.snapshot();
        Assert.assertTrue(snapshot.getFallbackCandidates().size() > 0);
        var local =  snapshot.getNearest().getAddress().getHost();
        Assert.assertTrue(snapshot.getFallbackCandidates()
                .stream()
                .noneMatch(transport -> local.equals(transport.getAddress().getHost())));
        snapshot.getFallbackCandidates().stream()
                .map(FakeTransport::getAddress)
                .map(HostAndPort::getHost)
                .forEach(host -> Assert.assertTrue(host + " is not from same dc as " + selfFqdn, host.contains("-sas-")));
    }

    @Test
    public void changedMapping() {
        nearestList.rearrange().join();

        String self = "alerting-vla-02.mon.yandex.net";
        resolver.put(selfFqdn, STATIC_DNS_CACHE.get(self));

        nearestList.rearrange().join();

        var snapshot = nearestList.snapshot();
        Assert.assertTrue(snapshot.getFallbackCandidates().size() > 0);
        var local =  snapshot.getNearest().getAddress().getHost();
        Assert.assertEquals(self.replace("alerting", "gateway"), snapshot.getNearest().getAddress().getHost());
        Assert.assertTrue(snapshot.getFallbackCandidates()
                .stream()
                .noneMatch(transport -> local.equals(transport.getAddress().getHost())));
        snapshot.getFallbackCandidates().stream()
                .map(FakeTransport::getAddress)
                .map(HostAndPort::getHost)
                .forEach(host -> Assert.assertTrue(host + " is not from same dc as " + self, host.contains("-vla-")));
    }

    @Test
    public void resolverSelfFailed() {
        nearestList.rearrange().join();

        resolver.put(selfFqdn, null);

        nearestList.rearrange().join();

        var snapshot = nearestList.snapshot();
        Assert.assertTrue(snapshot.getFallbackCandidates().size() > 0);
        var local =  snapshot.getNearest().getAddress().getHost();
        Assert.assertNotNull(snapshot.getNearest());
        Assert.assertTrue(snapshot.getFallbackCandidates()
                .stream()
                .noneMatch(transport -> local.equals(transport.getAddress().getHost())));
        Assert.assertEquals(8, snapshot.getFallbackCandidates().size());
    }

    @Test
    public void resolverAllFailed() {
        nearestList.rearrange().join();

        STATIC_DNS_CACHE.keySet().forEach(host -> resolver.put(host, null));

        nearestList.rearrange().join();

        var snapshot = nearestList.snapshot();
        Assert.assertTrue(snapshot.getFallbackCandidates().size() > 0);
        var local =  snapshot.getNearest().getAddress().getHost();
        Assert.assertNotNull(snapshot.getNearest());
        Assert.assertTrue(snapshot.getFallbackCandidates()
                .stream()
                .noneMatch(transport -> local.equals(transport.getAddress().getHost())));
        Assert.assertEquals(8, snapshot.getFallbackCandidates().size());
    }

    @Test
    public void deadResolverRelyOnShuffle() {
        DnsResolver deadResolver = (ignore) -> CompletableFuture.completedFuture(null);

        var nearestLists = ALERTING_HOSTS.stream()
                .map(host -> new NearestList<FakeTransport>(host, deadResolver))
                .collect(Collectors.toList());

        for (var nearestList : nearestLists) {
            for (var host : DATAPROXY_HOSTS) {
                nearestList.add(new FakeTransport(HostAndPort.fromParts(host, 5770)));
            }
            nearestList.rearrange().join();
        }

        List<String> localDps = nearestLists.stream()
                .map(NearestList::snapshot)
                .map(NearestList.Snapshot::getNearest)
                .map(FakeTransport::getAddress)
                .map(HostAndPort::getHost)
                .collect(Collectors.toList());

        Assert.assertTrue("Load is not balanced", new HashSet<>(localDps).size() > 0.7 * DATAPROXY_HOSTS.size());

        Map<String, String> alertingToDataproxy = new HashMap<>();
        for (int i = 0; i < ALERTING_HOSTS.size(); i++) {
            alertingToDataproxy.put(ALERTING_HOSTS.get(i), localDps.get(i));
        }

        for (int iter = 0; iter < 10; iter++) {
            for (var nearestList : nearestLists) {
                nearestList.rearrange().join();
            }

            for (int i = 0; i < ALERTING_HOSTS.size(); i++) {
                Assert.assertEquals("nearest host changed",
                        alertingToDataproxy.get(ALERTING_HOSTS.get(i)),
                        nearestLists.get(i).snapshot().getNearest().getAddress().getHost());
            }
        }
    }

    private static Inet6Address fromBytes(int... bytes) {
        try {
            byte[] b = new byte[bytes.length];
            for (int i = 0; i < bytes.length; i++) {
                b[i] = (byte) bytes[i];
            }
            return (Inet6Address) InetAddress.getByAddress(b);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    // don't actually do networking in test
    private static final Map<String, Inet6Address> STATIC_DNS_CACHE = new HashMap<>();

    static {
        STATIC_DNS_CACHE.put("gateway-myt-00.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,    0,   56, -114,    0,    0,   76,  125,  -38,  -19,   51,   52));
        STATIC_DNS_CACHE.put("gateway-vla-00.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   24,    4,   34,    0,    0,   76,  125,   83,   43,  109,  106));
        STATIC_DNS_CACHE.put("gateway-sas-01.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   22,   39, -105,    0,    0,   76,  125,   30,  -15,  -28,   42));
        STATIC_DNS_CACHE.put("gateway-vla-02.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   24,   10, -118,    0,    0,   76,  125,   63,   60,  -36,  112));
        STATIC_DNS_CACHE.put("gateway-myt-02.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,    0,   49, -114,    0,    0,   76,  125,  -24,  -97,  -28,   91));
        STATIC_DNS_CACHE.put("gateway-vla-01.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   24,    6,  -96,    0,    0,   76,  125,  -27,   46,   39,  -19));
        STATIC_DNS_CACHE.put("gateway-sas-02.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   22,   39,    5,    0,    0,   76,  125,  -48, -102,   31,   68));
        STATIC_DNS_CACHE.put("gateway-myt-01.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,    0,   56,   28,    0,    0,   76,  125,  -59,  115,   46,   48));
        STATIC_DNS_CACHE.put("gateway-sas-00.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   22,   41,   21,    0,    0,   76,  125,   52,   12, -125,   -4));
        STATIC_DNS_CACHE.put("alerting-myt-00.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,    0,   56, -114,    0,    0,   76,  125,  -26,  117,    0,  125));
        STATIC_DNS_CACHE.put("alerting-vla-00.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   24,    4,   34,    0,    0,   76,  125,  -99,  -96,   29,   18));
        STATIC_DNS_CACHE.put("alerting-sas-01.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   22,   39, -105,    0,    0,   76,  125, -127,  111,  -65,  -71));
        STATIC_DNS_CACHE.put("alerting-vla-02.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   24,   10, -118,    0,    0,   76,  125,  105,   28,  -71,   83));
        STATIC_DNS_CACHE.put("alerting-myt-02.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,    0,   49, -114,    0,    0,   76,  125,  -23,  -16,   57,   51));
        STATIC_DNS_CACHE.put("alerting-vla-01.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   24,    6,  -96,    0,    0,   76,  125, -102,  -23,   32,   90));
        STATIC_DNS_CACHE.put("alerting-sas-02.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   22,   39,    5,    0,    0,   76,  125,   -1,  -53,   -6,   27));
        STATIC_DNS_CACHE.put("alerting-myt-01.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,    0,   56,   28,    0,    0,   76,  125,   70,  -20,  -34,   99));
        STATIC_DNS_CACHE.put("alerting-sas-00.mon.yandex.net", fromBytes(  42,    2,    6,  -72,   12,   22,   41,   21,    0,    0,   76,  125,   30,  -81,   89,   63));
    }

    public static void main(String[] args) {
        var resolver = new DefaultDnsResolver();
        for (var hosts : List.of(DATAPROXY_HOSTS, ALERTING_HOSTS)) {
            for (var host : hosts) {
                var bytes = resolver.resolve(host).join().getAddress();
                System.out.println("STATIC_DNS_CACHE.put(\"" + host + "\", " + IntStream.range(0, bytes.length)
                        .mapToObj(i -> String.format("% 4d", (int) bytes[i]))
                        .collect(Collectors.joining(", ", "fromBytes(", "));")));
            }
        }
    }
}
