package ru.yandex.discovery;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;

import com.google.common.net.HostAndPort;

import ru.yandex.discovery.conductor.ConductorResolveService;
import ru.yandex.discovery.k8s.K8SResolveService;
import ru.yandex.discovery.solomon.SolomonResolverService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.SolomonEnv;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.toList;

/**
 * @author alexlovkov
 */
final class AsyncDiscoveryService implements DiscoveryService {

    private final ResolveService[] resolvers;

    public AsyncDiscoveryService(SolomonEnv env, Executor executor) {
        resolvers = new ResolveService[] {
            new ConductorResolveService(executor),
            new SolomonResolverService(env, executor),
            new K8SResolveService(executor)
        };
    }

    public CompletableFuture<List<HostAndPort>> resolve(List<String> hosts) {
        return hosts.stream()
            .map(this::resolve)
            .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(x ->
                x.stream()
                    .flatMap(Collection::stream)
                    .distinct()
                    .collect(Collectors.toList()));
    }

    private CompletableFuture<List<HostAndPort>> resolve(String host) {
        for (ResolveService resolver : resolvers) {
            if (host.startsWith(resolver.prefix())) {
                return resolver.resolve(host);
            }
        }
        return completedFuture(Collections.singletonList(HostAndPort.fromString(host)));
    }

    static final class Holder {
        static final AsyncDiscoveryService INSTANCE =
                new AsyncDiscoveryService(SolomonEnv.current(), ForkJoinPool.commonPool());
    }
}
