package ru.yandex.discovery.cluster;

import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.net.HostAndPort;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.config.protobuf.TClusterConfig;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class ClusterMapperImpl implements ClusterMapper, AutoCloseable {
    private final DiscoveryService discovery;
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private final CompletableFuture<Void> init = new CompletableFuture<>();

    private final ActorRunner actor;
    private final Map<String, Cluster> clusterById;
    private volatile Map<String, String> clusterByHost = new HashMap<>();

    public ClusterMapperImpl(List<TClusterConfig> clusters, DiscoveryService discovery, Executor executor, ScheduledExecutorService timer) {
        this.discovery = discovery;
        this.executor = executor;
        this.timer = timer;

        var clusterById = new HashMap<String, Cluster>();
        for (var config : clusters) {
            final String clusterId = config.getClusterId();
            if (clusterById.containsKey(clusterId)) {
                throw new IllegalStateException("non unique clusterId: " + clusterId);
            }

            clusterById.put(clusterId, new Cluster(config));
        }

        this.clusterById = Map.copyOf(clusterById);
        this.actor = new ActorRunner(this::act, executor);
        forcePing();
        if (!this.clusterById.isEmpty()) {
            init.join();
        }
    }

    private void act() {
        boolean initDone = true;
        var clusterByHost = new HashMap<String, String>();
        for (var entry : clusterById.entrySet()) {
            initDone &= entry.getValue().init.get();

            var clusterId = entry.getKey();
            var hosts = entry.getValue().hosts;
            for (var host : hosts) {
                var prev = clusterByHost.put(host, clusterId);
                if (prev != null) {
                    String msg = "host " + host +
                            " in multiple clusters [" + prev + ", " + clusterId + ']';
                    throw new IllegalArgumentException(msg);
                }
            }
        }
        this.clusterByHost = clusterByHost;
        if (initDone) {
            init.complete(null);
        }
    }

    @Nullable
    @Override
    public String byFqdnOrNull(String fqdn) {
        if (StringUtils.isEmpty(fqdn)) {
            return null;
        }

        var result = clusterByHost.get(fqdn.toLowerCase());
        if (result == null) {
            forcePing();
        }

        return result;
    }

    @Nullable
    @Override
    public String byParamOrNull(String param) {
        if (StringUtils.isEmpty(param)) {
            return null;
        }

        String lowerParam = param.toLowerCase();
        return clusterById.containsKey(lowerParam) ? lowerParam : null;
    }

    @Override
    public Set<String> knownClusterIds() {
        return Collections.unmodifiableSet(clusterById.keySet());
    }

    @Override
    public String byFqdnOrThrow(String fqdn) {
        String cluster = byFqdnOrNull(fqdn);
        if (cluster == null) {
            throw new IllegalArgumentException("cannot determine cluster by fqdn: " + fqdn);
        }
        return cluster;
    }

    private void forcePing() {
        for (var cluster : clusterById.values()) {
            cluster.forcePing();
        }
    }

    @Override
    public void close() throws Exception {
        for (var cluster : clusterById.values()) {
            cluster.close();
        }
    }

    private class Cluster implements AutoCloseable {
        private final TClusterConfig config;
        private final PingActorRunner actor;
        private volatile Set<String> hosts = Set.of();
        private final AtomicBoolean init = new AtomicBoolean();

        public Cluster(TClusterConfig config) {
            this.config = config;
            this.actor = PingActorRunner.newBuilder()
                    .onPing(this::update)
                    .executor(executor)
                    .timer(timer)
                    .pingInterval(Duration.ofMinutes(60))
                    .build();
        }

        public void forcePing() {
            actor.forcePing();
        }

        private CompletableFuture<?> update(int attempt) {
            return discovery.resolve(config.getAddressesList())
                    .thenAccept(hosts -> {
                        this.hosts = hosts.stream()
                                .map(HostAndPort::getHost)
                                .collect(Collectors.toSet());
                        init.set(true);
                        ClusterMapperImpl.this.actor.schedule();
                    });
        }

        @Override
        public void close() {
            this.actor.close();
        }
    }
}
