package ru.yandex.discovery.conductor;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse.BodyHandlers;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * Simple synchronous conductor client with in-memory cache.
 *
 * @author Sergey Polovko
 */
final class ConductorClient {
    private final HttpClient httpClient;
    private final Cache<String, List<String>> cache = CacheBuilder.newBuilder()
            .expireAfterWrite(1, TimeUnit.MINUTES)
            .build();

    public ConductorClient(Executor executor) {
        this.httpClient = HttpClient.newBuilder()
            .executor(executor)
            .followRedirects(HttpClient.Redirect.NEVER)
            .connectTimeout(Duration.ofSeconds(5))
            .version(HttpClient.Version.HTTP_1_1)
            .build();
    }

    CompletableFuture<List<String>> groupToHosts(String group) {
        List<String> hosts = cache.getIfPresent(group);
        if (hosts != null) {
            return completedFuture(hosts);
        }

        var request = HttpRequest.newBuilder(URI.create("http://c.yandex-team.ru/api/groups2hosts/" + group))
            .timeout(Duration.ofSeconds(10))
            .build();

        return httpClient.sendAsync(request, BodyHandlers.ofLines())
            .thenApply(response -> {
                if (response.statusCode() != 200) {
                    String message = String.format(
                        "cannot resolve conductor group %s, response status: %d",
                        group, response.statusCode());
                    throw new IllegalStateException(message);
                }

                var newHosts = response.body().collect(Collectors.toList());
                cache.put(group, newHosts);
                return newHosts;
            });
    }
}
