package ru.yandex.discovery.conductor;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import javax.annotation.CheckReturnValue;

import com.google.common.net.HostAndPort;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.discovery.ResolveService;
import ru.yandex.misc.io.RuntimeIoException;

import static java.nio.file.StandardOpenOption.CREATE;

/**
 * @author alexlovkov
 */
public class ConductorResolveService implements ResolveService {

    private static final Logger logger = LoggerFactory.getLogger(ConductorResolveService.class);
    private static final String FOLDER_FOR_CACHE_FILES = "conductor_cache/";
    private static final String CONDUCTOR_PREFIX = "conductor_group://";

    private final ConductorClient client;
    private final String cacheDir;

    public ConductorResolveService(Executor executor, String cacheDir) {
        this.client = new ConductorClient(executor);
        this.cacheDir = cacheDir;
    }

    public ConductorResolveService(Executor executor) {
        this(executor, System.getProperty("user.dir"));
    }

    @Override
    public String prefix() {
        return CONDUCTOR_PREFIX;
    }

    /**
     * return hosts by conductor group
     *
     * @param conductorGroup name of the group, with optional port divided by :
     * @return lists of address by conductor group. port in address is zero if it wasn't given in conductor group.
     */
    @Override
    public CompletableFuture<List<HostAndPort>> resolve(String conductorGroup) {
        if (!isValidConductorGroup(conductorGroup)) {
            CompletableFuture f = new CompletableFuture();
            f.completeExceptionally(new IllegalArgumentException("conductorGroup doesn't start with prefix: " + CONDUCTOR_PREFIX));
            return f;
        }
        String group = conductorGroup.substring(CONDUCTOR_PREFIX.length());

        final HostAndPort groupHostAndPort = HostAndPort.fromString(group);
        final CompletableFuture<List<String>> hosts = client.groupToHosts(groupHostAndPort.getHost());
        CompletableFuture<List<HostAndPort>> addresses = new CompletableFuture<>();
        hosts.whenCompleteAsync((r, x) -> {
            String groupName = groupHostAndPort.getHost();
            if (x == null) {
                addresses.complete(mapStringsToHostAndPortes(r, groupHostAndPort));

                try {
                    writeHostsToFile(groupName, r, addresses);
                } catch (Exception e) {
                    logger.error("can't write conductor cache to file {}", Path.of(FOLDER_FOR_CACHE_FILES, groupName), e);
                }
            } else {
                // normal way, try use cache
                logger.warn("can't get hosts of conductor group {}", groupName, x);
                try {
                    addresses.complete(mapStringsToHostAndPortes(getHostsFromFile(groupName), groupHostAndPort));
                } catch (IOException e) {
                    x.addSuppressed(new RuntimeIoException("can't read hosts from file", e));
                    addresses.completeExceptionally(x);
                }
            }
        });
        return addresses;
    }

    @CheckReturnValue
    private boolean isValidConductorGroup(String conductorGroup) {
        return conductorGroup.startsWith(CONDUCTOR_PREFIX);
    }

    private List<HostAndPort> mapStringsToHostAndPortes(List<String> list, HostAndPort group) {
        var result = new ArrayList<HostAndPort>(list.size());
        for (String host : list) {
            if (group.hasPort()) {
                result.add(HostAndPort.fromParts(host, group.getPort()));
            } else {
                result.add(HostAndPort.fromHost(host));
            }
        }
        return result;
    }

    private List<String> getHostsFromFile(String name) throws IOException {
        logger.info("try to get hosts from cached file: {}", name);
        Path file = Paths.get(cacheDir, FOLDER_FOR_CACHE_FILES, name + ".txt");
        return Files.readAllLines(file);
    }

    private void writeHostsToFile(
        String name,
        List<String> hosts,
        CompletableFuture<List<HostAndPort>> addresses) throws IOException
    {
        try {
            Files.createDirectory(Paths.get(cacheDir, FOLDER_FOR_CACHE_FILES));
        } catch (FileAlreadyExistsException ignored) {
        }
        Path file = Paths.get(cacheDir, FOLDER_FOR_CACHE_FILES, name + ".txt");
        try {
            Files.write(file, hosts, StandardCharsets.UTF_8, CREATE);
        } catch (IOException e) {
            addresses.completeExceptionally(new RuntimeIoException("can't write hosts to the file: " + name, e));
        }
    }
}
