package ru.yandex.discovery.k8s;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse.BodyHandlers;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.security.KeyStore;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.net.HostAndPort;

import ru.yandex.discovery.ResolveService;

/**
 * @author Vladimir Gordiychuk
 */
public class K8SResolveService implements ResolveService {
    private static final String PREFIX = "k8s://";
    private static final Path CLUSTER_CONFIG = Path.of("/etc/kubernetes/info/cluster.json");

    private final Executor executor;
    private final ObjectMapper mapper = new ObjectMapper();

    private final AtomicReference<Cluster> cluster = new AtomicReference<>();
    private final AtomicReference<Certificate> certificate = new AtomicReference<>();
    private final AtomicReference<HttpClient> httpClient = new AtomicReference<>();

    public K8SResolveService(Executor executor) {
        this.executor = executor;
    }

    @Override
    public String prefix() {
        return PREFIX;
    }

    @Override
    public CompletableFuture<List<HostAndPort>> resolve(String str) {
        if (!str.startsWith(PREFIX)) {
           return CompletableFuture.failedFuture(new IllegalArgumentException("unsupported format for k8s: " + str));
        }

        var request = Request.of(str);

        ensureClusterFresh();
        ensureCertificateFresh();

        var cluster = this.cluster.get();
        var uri = URI.create(cluster.master + "/api/v1/namespaces/" + readFile(cluster.namespace) + "/endpoints/" + request.service + "-service");
        var http = getHttpClient();

        var req = HttpRequest.newBuilder()
                .header("Authorization", "Bearer "+ readFile(cluster.token))
                .header("Accept", "application/json")
                .uri(uri)
                .build();

        return http.sendAsync(req, BodyHandlers.ofString())
                .thenApply(response -> {
                    if (response.statusCode() != 200) {
                        String message = String.format(
                                "cannot resolve k8s service endpoints %s, response status: %d, body: %s", str, response.statusCode(), response.body());
                        throw new IllegalStateException(message);
                    }

                    return parseResponse(response.body(), request.port);
                });
    }

    private List<HostAndPort> parseResponse(String body, int port) {
        try {
            var result = new ArrayList<HostAndPort>();
            var root = mapper.readTree(body);
            var subsets = root.get("subsets");
            for (var subset : subsets) {
                var addresses = subset.get("addresses");
                for (var address : addresses) {
                    var ip = address.get("ip").asText();
                    result.add(HostAndPort.fromParts(ip, port));
                }
            }

            return result;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void ensureClusterFresh() {
        var prev = cluster.get();
        if (prev == null || prev.time.compareTo(lastModifiedAt(CLUSTER_CONFIG)) != 0) {
            cluster.compareAndSet(prev, readCluster(CLUSTER_CONFIG));
        }
    }

    private void ensureCertificateFresh() {
        var cluster = this.cluster.get();
        var prev = certificate.get();
        if (prev == null || !prev.path.equals(cluster.certificate) || prev.time.compareTo(lastModifiedAt(cluster.certificate)) != 0) {
            if (certificate.compareAndSet(prev, readCertificate(cluster.certificate))) {
                httpClient.set(null);
            }
        }
    }

    private HttpClient getHttpClient() {
        HttpClient prev;
        HttpClient update;
        do {
            prev = this.httpClient.get();
            if (prev != null) {
                return prev;
            }

            update = HttpClient.newBuilder()
                    .executor(executor)
                    .connectTimeout(Duration.ofSeconds(5))
                    .followRedirects(HttpClient.Redirect.NEVER)
                    .version(HttpClient.Version.HTTP_1_1)
                    .sslContext(certificate.get().context())
                    .build();
        } while (!this.httpClient.compareAndSet(null, update));
        return update;
    }

    public static String read(String path) throws IOException {
        return Files.readString(Path.of(path), StandardCharsets.UTF_8).trim();
    }

    public static String readFile(Path path) {
        try {
            return Files.readString(path, StandardCharsets.UTF_8).trim();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Cluster readCluster(Path path) {
        try {
            FileTime time = Files.getLastModifiedTime(path);
            ObjectMapper mapper = new ObjectMapper();
            JsonNode root = mapper.readTree(readFile(path));
            String master = root.get("master").asText();
            Path token = Path.of(root.get("token-path").asText());
            Path namespace = Path.of(root.get("namespace-path").asText());
            Path certificate = Path.of(root.get("CA-certificate-path").asText());
            return new Cluster(master, token, namespace, certificate, time);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    public static FileTime lastModifiedAt(Path path) {
        try {
            return Files.getLastModifiedTime(path);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Certificate readCertificate(Path path) {
        try (var input = Files.newInputStream(path)) {
            var time = Files.getLastModifiedTime(path);
            var cf = CertificateFactory.getInstance("X.509");
            var caCert = (X509Certificate) cf.generateCertificate(input);

            var tmf = TrustManagerFactory
                    .getInstance(TrustManagerFactory.getDefaultAlgorithm());

            var ks = KeyStore.getInstance(KeyStore.getDefaultType());
            ks.load(null);
            ks.setCertificateEntry("caCert", caCert);

            tmf.init(ks);

            var sslContext = SSLContext.getInstance("TLS");
            sslContext.init(null, tmf.getTrustManagers(), null);
            return new Certificate(sslContext, path, time);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public record Request(String service, int port) {
        public static Request of(String str) {
            var hostAndPort = HostAndPort.fromString(str.substring(PREFIX.length()));
            return new Request(hostAndPort.getHost(), hostAndPort.getPort());
        }
    }

    public record Cluster(String master, Path token, Path namespace, Path certificate, FileTime time) {
    }

    public record Certificate(SSLContext context, Path path, FileTime time) {
    }
}
