package ru.yandex.discovery.solomon;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Charsets;
import com.google.common.net.HostAndPort;

import ru.yandex.discovery.ResolveService;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.solomon.util.SolomonEnv;


/**
 * @author alexlovkov
 **/
public class SolomonResolverService implements ResolveService {

    private static final Logger logger = LoggerFactory.getLogger(SolomonResolverService.class);
    private static final String SOLOMON_PREFIX = "solomon://";

    private final HttpClient httpClient;
    private final String envType;
    private final ObjectMapper objectMapper;

    public SolomonResolverService(SolomonEnv env, Executor executor) {
        this.httpClient = HttpClient.newBuilder()
            .executor(executor)
            .followRedirects(HttpClient.Redirect.NEVER)
            .connectTimeout(Duration.ofSeconds(5))
            .version(HttpClient.Version.HTTP_1_1)
            .build();
        this.envType = env.name();
        this.objectMapper = new ObjectMapper();
    }

    @Override
    public String prefix() {
        return SOLOMON_PREFIX;
    }

    @Override
    public CompletableFuture<List<HostAndPort>> resolve(String solomonAddress) {
        logger.debug("resolving {}", solomonAddress);
        URI uri = URI.create("https://" + solomonAddress.substring(SOLOMON_PREFIX.length()));
        var request = HttpRequest.newBuilder(uri)
            .timeout(Duration.ofSeconds(10))
            .build();
        return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray())
            .thenApply(r -> {
                if (r.statusCode() != 200) {
                    String message = String.format("cannot resolve %s, code: %d", solomonAddress, r.statusCode());
                    throw new IllegalStateException(message);
                }
                try {
                    var response = objectMapper.readValue(r.body(), SolomonDiscoveryResponse.class);
                    if (response.getPort().getNum() != -1) {
                        return response.getHosts().stream()
                                .map(e -> HostAndPort.fromParts(e.getFqdn(), response.getPort().getNum()))
                                .distinct()
                                .collect(Collectors.toList());
                    } else {
                        return response.getHosts().stream()
                                .map(e -> HostAndPort.fromHost(e.getFqdn()))
                                .distinct()
                                .collect(Collectors.toList());
                    }
                } catch (IOException e) {
                    String message = String.format(
                        "can't parse json from solomonDiscovery: %s, solomonAddress: %s",
                        new String(r.body(), Charsets.UTF_8), solomonAddress);
                    throw new RuntimeException(message, e);
                }
            });
    }
}
