package ru.yandex.discovery.conductor;

import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.stream.IntStream;

import com.google.common.net.HostAndPort;
import com.google.common.util.concurrent.MoreExecutors;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.rules.TestName;

import static org.junit.Assert.assertNotNull;

/**
 * @author alexlovkov
 */
public class ConductorResolveServiceTest {

    @Rule
    public TemporaryFolder tmp = new TemporaryFolder();
    @Rule
    public TestName testName = new TestName();

    private ConductorResolveService resolver;

    @Before
    public void setUp() throws Exception {
        this.resolver = new ConductorResolveService(
                MoreExecutors.directExecutor(),
                tmp.newFolder(testName.getMethodName()).toString());
    }

    @Test(expected = IllegalArgumentException.class)
    public void incorrectConductorGroupTest() throws Throwable {
        final Throwable[] exception = new Throwable[1];
        resolver.resolve("QWE123")
            .exceptionally(e -> {
                exception[0] = e;
                return null;
            }).get();
        throw exception[0];
    }

    @Test(expected = IllegalArgumentException.class)
    public void incorrectPrefixConductorGroupTest() throws Throwable {
        final Throwable[] exception = new Throwable[1];
        resolver.resolve("conductor_Qgroup://solomon_prod_data_storage_man")
            .exceptionally(e -> {
                exception[0] = e;
                return null;
            }).get();
        throw exception[0];
    }

    @Test
    public void notExpectedParseErrorTest() throws ExecutionException, InterruptedException {
        try {
            resolver.resolve("conductor_group://solomon_prod_storage_vla").get();
        } catch (IllegalArgumentException e) {
            Assert.fail();
        }
    }

    @Test
    public void hostsWithPort() throws ExecutionException, InterruptedException {
        List<HostAndPort> addresses = resolver.resolve("conductor_group://solomon_prod_storage_sas:666").get();
        validateHostsAndPort(addresses, 666);
    }

    @Test
    public void hostsWithoutPort() throws ExecutionException, InterruptedException {
        List<HostAndPort> addresses = resolver.resolve("conductor_group://solomon_prod_storage_sas").get();
        var expected = IntStream.range(0, addresses.size())
                .mapToObj(idx -> HostAndPort.fromHost(String.format("solomon-storage-sas-%03d.search.yandex.net", idx)))
                .toArray(HostAndPort[]::new);
        Assert.assertArrayEquals(expected, addresses.toArray());
    }

    @Test
    public void unknownConductorGroupTest() throws Throwable {
        var e = resolver.resolve("conductor_group://QWEsolomon_prod_data_storage_man")
                .thenApply(ignore -> (Throwable) null)
                .exceptionally(throwable -> throwable)
                .join();

        assertNotNull(e);
    }

    private void validateHostsAndPort(List<HostAndPort> addresses, int expectedPort) {
        for (int i = 0; i < addresses.size(); i++) {
            HostAndPort address = addresses.get(i);
            String host = String.format("solomon-storage-sas-%03d.search.yandex.net", i);
            Assert.assertEquals(host, address.getHost());
        }
        addresses.forEach(e -> {
            Assert.assertEquals(expectedPort, e.getPort());
        });
    }
}
