package ru.yandex.solomon.locks;

import java.time.Clock;
import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DistributedLockImpl implements DistributedLock {
    private static final Logger logger = LoggerFactory.getLogger(DistributedLockImpl.class);
    private static final long DEFAULT_WATCH_DELAY_MILLIS = TimeUnit.SECONDS.toMillis(1);

    private final String lockId;
    private final String me;
    private final LocksDao dao;
    private final Clock clock;
    private final ScheduledExecutorService executor;

    private final AtomicReference<State> state;
    private final DistributedLockMetrics metrics;

    public DistributedLockImpl(
            String lockId,
            String lockAlias,
            String me,
            LocksDao dao,
            Clock clock,
            ScheduledExecutorService executor,
            MetricRegistry registry)
    {
        this.lockId = lockId;
        this.me = me;
        this.dao = dao;
        this.clock = clock;
        this.executor = executor;
        this.metrics = new DistributedLockMetrics(lockAlias, registry);
        this.state = new AtomicReference<>(new IdleState());
    }

    @Override
    public boolean isLockedByMe() {
        LockDetail detail = this.state.get().getLockDetail();
        return detail != null && me.equals(detail.owner());
    }

    @Override
    public Optional<LockDetail> lockDetail() {
        return Optional.ofNullable(this.state.get().getLockDetail());
    }

    @Override
    public CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo) {
        return this.state.get().getLockDetail(latestVisibleSeqNo);
    }

    @Override
    public void acquireLock(LockSubscriber subscription, long leaseTime, TimeUnit unit) {
        if (subscription.isCanceled()) {
            throw new IllegalArgumentException("LockSubscriber already canceled: " + subscription);
        }
        long leaseMillis = unit.toMillis(leaseTime);
        this.state.get().acquireLock(subscription, leaseMillis);
    }

    @Override
    public CompletableFuture<Boolean> unlock() {
        return this.state.get().unlock();
    }

    private boolean changeState(State current, State next) {
        if (state.compareAndSet(current, next)) {
            if (logger.isDebugEnabled()) {
                logger.debug("{}/{} change state {} -> {}", lockId, me, current, next);
            }
            next.run();
            return true;
        }

        return false;
    }

    private boolean inState(State current) {
        return Objects.equals(current, state.get());
    }

    private long randomizeDelay(long delayMillis) {
        return DurationUtils.randomize(delayMillis);
    }

    private long timeToExpire(LockDetail details) {
        return Math.max(details.expiredAt().toEpochMilli() - clock.millis(), DEFAULT_WATCH_DELAY_MILLIS);
    }

    @ParametersAreNonnullByDefault
    public abstract static class State implements Runnable {
        @Nullable
        public abstract LockDetail getLockDetail();

        public abstract CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo);

        public abstract void acquireLock(LockSubscriber subscription, long leaseLockMillis);

        public abstract CompletableFuture<Boolean> unlock();

        public abstract void cancel();

        @Override
        public String toString() {
            return this.getClass().getSimpleName();
        }
    }

    /**
     * Retry until target lock will not acquired
     */
    @ParametersAreNonnullByDefault
    private class AcquireState extends State {
        private final LockSubscriber subscription;
        private final long leaseLockMillis;
        private AtomicReference<LockDetail> detail = new AtomicReference<>();
        @Nullable
        private volatile Future future;

        private AcquireState(LockSubscriber subscription, long leaseLockMillis) {
            this.subscription = subscription;
            this.leaseLockMillis = leaseLockMillis;
        }

        @Override
        public void run() {
            scheduleRetry(0);
        }

        private void tryAcquire() {
            if (ensureNotCanceled()) {
                return;
            }

            Instant expiredAt = clock.instant().plusMillis(leaseLockMillis);
            future = dao.acquireLock(lockId, me, expiredAt)
                    .whenComplete((details, e) -> {
                        if (ensureNotCanceled()) {
                            return;
                        }

                        if (e != null) {
                            logger.error("{}/{} acquire lock failed", lockId, me, e);
                            scheduleRetry(DEFAULT_WATCH_DELAY_MILLIS);
                            return;
                        }

                        if (!Objects.equals(details.owner(), me)) {
                            this.detail.set(details);
                            scheduleRetry(timeToExpire(details));
                            return;
                        }

                        changeState(this, new ExtendState(subscription, leaseLockMillis, details));
                    });
        }

        private boolean ensureNotCanceled() {
            if (!inState(this)) {
                return false;
            }

            if (subscription.isCanceled()) {
                changeState(this, new WatchState());
                return true;
            }

            return false;
        }

        private void scheduleRetry(long delayMillis) {
            delayMillis = randomizeDelay(delayMillis);
            logger.debug("{}/{} schedule acquire after {} ms", lockId, me, delayMillis);
            future = executor.schedule(this::tryAcquire, delayMillis, TimeUnit.MILLISECONDS);
        }

        @Nullable
        @Override
        public LockDetail getLockDetail() {
            return detail.get();
        }

        @Override
        public CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo) {
            LockDetail copy = detail.get();
            if (copy != null && Long.compareUnsigned(copy.seqNo(), latestVisibleSeqNo) >= 0) {
                return CompletableFuture.completedFuture(Optional.of(copy));
            }

            return dao.readLock(lockId)
                    .whenComplete((r, e) -> {
                        if (e != null) {
                            return;
                        }

                        r.ifPresent(lockDetail -> detail.compareAndSet(copy, lockDetail));
                    });
        }

        @Override
        public void acquireLock(LockSubscriber subscription, long leaseLockMillis) {
            if (!this.subscription.isCanceled()) {
                throw new IllegalArgumentException("Until previous LockSubscriber not canceled "
                        + this.subscription
                        + " not able add new subscription "
                        + subscription);
            }

            AcquireState acquireState = new AcquireState(subscription, leaseLockMillis);
            if (changeState(this, acquireState)) {
                cancel();
            } else {
                state.get().acquireLock(subscription, leaseLockMillis);
            }
        }

        @Override
        public CompletableFuture<Boolean> unlock() {
            cancel();
            WatchState watch = new WatchState();
            if (!changeState(this, watch)) {
                return state.get().unlock();
            }

            return CompletableFuture.completedFuture(false);
        }

        @Override
        public void cancel() {
            Future future = this.future;
            if (future != null) {
                future.cancel(true);
            }
        }
    }

    /**
     * Periodically extend lease lock time
     */
    @ParametersAreNonnullByDefault
    private class ExtendState extends State {
        private final LockSubscriber subscription;
        private final long leaseLockMillis;
        private final AtomicReference<LockDetail> detail = new AtomicReference<>();
        @Nullable
        private volatile Future future;

        public ExtendState(LockSubscriber subscription, long leaseLockMillis, LockDetail detail) {
            this.subscription = subscription;
            this.leaseLockMillis = leaseLockMillis;
            this.detail.set(detail);
        }

        @Override
        public void run() {
            metrics.lockAcquired();
            scheduleExtendLock();
            subscription.onLock(detail.get().seqNo());
        }

        private void scheduleExtendLock() {
            long delayMs = DurationUtils.randomize(Math.min(leaseLockMillis / 3, 1_000));
            logger.debug("{}/{} schedule extend lock after {} ms", lockId, me, delayMs);
            this.future = executor.schedule(this::runScheduledExtend, delayMs, TimeUnit.MILLISECONDS);
        }

        private void runScheduledExtend() {
            if (ensureNotCanceled()) {
                return;
            }

            Instant expiredAt = Instant.ofEpochMilli(clock.millis() + leaseLockMillis);
            future = dao.extendLockTime(lockId, me, expiredAt)
                    .whenComplete((success, e) -> {
                        if (ensureNotCanceled()) {
                            return;
                        }

                        if (e != null) {
                            logger.error("{}/{} failed extend lock", lockId, me, e);
                            if (clock.millis() >= detail.get().expiredAt().toEpochMilli()) {
                                moveToWatch(UnlockReason.CONNECTION_LOST);
                                return;
                            }

                            executor.execute(this::runScheduledExtend);
                            return;
                        }

                        if (!success) {
                            logger.debug("{}/{} lease expired for lock", lockId, me);
                            moveToWatch(UnlockReason.LEASE_EXPIRED);
                            return;
                        }

                        LockDetail copy = this.detail.get();
                        this.detail.compareAndSet(copy, copy.expiredAt(expiredAt));
                        scheduleExtendLock();
                    });
        }

        private boolean ensureNotCanceled() {
            if (!inState(this)) {
                return false;
            }

            if (subscription.isCanceled()) {
                moveToWatch(UnlockReason.MANUAL_UNLOCK);
                return true;
            }

            return false;
        }

        private void moveToWatch(UnlockReason reason) {
            if (changeState(this, new WatchState())) {
                metrics.releaseLock(reason);
                subscription.onUnlock(reason);
            }
        }

        @Nullable
        @Override
        public LockDetail getLockDetail() {
            return detail.get();
        }

        @Override
        public CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo) {
            LockDetail copy = detail.get();
            if (copy != null && Long.compareUnsigned(copy.seqNo(), latestVisibleSeqNo) >= 0) {
                return CompletableFuture.completedFuture(Optional.of(copy));
            }

            return dao.readLock(lockId)
                    .whenComplete((r, e) -> {
                        if (e != null) {
                            return;
                        }

                        r.ifPresent(lockDetail -> detail.compareAndSet(copy, lockDetail));
                    });
        }

        @Override
        public void acquireLock(LockSubscriber subscription, long leaseLockMillis) {
            if (!this.subscription.isCanceled()) {
                throw new IllegalArgumentException("Until previous LockSubscriber not canceled "
                        + this.subscription
                        + " not able add new subscription "
                        + subscription);
            }

            AcquireState acquireState = new AcquireState(subscription, leaseLockMillis);
            if (changeState(this, acquireState)) {
                cancel();
            } else {
                state.get().acquireLock(subscription, leaseLockMillis);
            }
        }

        @Override
        public CompletableFuture<Boolean> unlock() {
            logger.debug("{}/{} manual unlock lock", lockId, me);
            cancel();
            WatchState watch = new WatchState(detail.get());
            if (!changeState(this, watch)) {
                metrics.releaseLock(UnlockReason.MANUAL_UNLOCK);
                return state.get().unlock();
            }

            return watch.unlock()
                    .handle((r, e) -> {
                        subscription.onUnlock(UnlockReason.MANUAL_UNLOCK);
                        if (e != null) {
                            throw Throwables.propagate(e);
                        }

                        return r;
                    });
        }

        @Override
        public void cancel() {
            Future future = this.future;
            if (future != null) {
                future.cancel(true);
            }
        }
    }

    /**
     * Track ownership for lock
     */
    @ParametersAreNonnullByDefault
    private class WatchState extends State {
        private final AtomicReference<LockDetail> detail = new AtomicReference<>();
        private volatile Future future;

        WatchState() {
        }

        WatchState(LockDetail detail) {
            this.detail.set(detail);
        }

        @Nullable
        @Override
        public LockDetail getLockDetail() {
            return detail.get();
        }

        @Override
        public CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo) {
            LockDetail copy = detail.get();
            if (copy != null && Long.compareUnsigned(copy.seqNo(), latestVisibleSeqNo) >= 0) {
                return CompletableFuture.completedFuture(Optional.of(copy));
            }

            return dao.readLock(lockId)
                    .whenComplete((r, e) -> {
                        if (e != null) {
                            return;
                        }

                        r.ifPresent(lockDetail -> detail.compareAndSet(copy, lockDetail));
                    });
        }

        @Override
        public void acquireLock(LockSubscriber subscription, long leaseLockMillis) {
            AcquireState acquireState = new AcquireState(subscription, leaseLockMillis);
            if (changeState(this, acquireState)) {
                cancel();
            } else {
                state.get().acquireLock(subscription, leaseLockMillis);
            }
        }

        @Override
        public CompletableFuture<Boolean> unlock() {
            return dao.releaseLock(lockId, me)
                    .thenApply(success -> {
                        if (success) {
                            detail.set(null);
                        }

                        return success;
                    });
        }

        @Override
        public void cancel() {
            Future future = this.future;
            if (future != null) {
                future.cancel(true);
            }
        }

        @Override
        public void run() {
            runWatch();
        }

        private void runWatch() {
            LockDetail prev = detail.get();
            future = dao.readLock(lockId)
                    .whenComplete((opt, e) -> {
                        if (e != null) {
                            logger.error("{}/{} failed read lock", lockId, me, e);
                            scheduleWatch(DEFAULT_WATCH_DELAY_MILLIS);
                            return;
                        }

                        if (opt.isEmpty()) {
                            this.detail.set(null);
                            scheduleWatch(DEFAULT_WATCH_DELAY_MILLIS);
                        } else {
                            LockDetail details = opt.get();
                            if (this.detail.compareAndSet(prev, details)) {
                                scheduleWatch(timeToExpire(details));
                            } else {
                                scheduleWatch(DEFAULT_WATCH_DELAY_MILLIS);
                            }
                        }
                    });
        }

        private void scheduleWatch(long delayMillis) {
            delayMillis = randomizeDelay(delayMillis);
            logger.debug("{}/{} schedule watch lock after {} ms", lockId, me, delayMillis);
            future = executor.schedule(this::runWatch, delayMillis, TimeUnit.MILLISECONDS);
        }
    }

    @ParametersAreNonnullByDefault
    private class IdleState extends State {

        @Nullable
        @Override
        public LockDetail getLockDetail() {
            changeState(this, new WatchState());
            return state.get().getLockDetail();
        }

        @Override
        public CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo) {
            changeState(this, new WatchState());
            return state.get().getLockDetail(latestVisibleSeqNo);
        }

        @Override
        public void acquireLock(LockSubscriber subscription, long leaseLockMillis) {
            AcquireState acquireState = new AcquireState(subscription, leaseLockMillis);
            if (!changeState(this, acquireState)) {
                state.get().acquireLock(subscription, leaseLockMillis);
            }
        }

        @Override
        public CompletableFuture<Boolean> unlock() {
            return CompletableFuture.completedFuture(Boolean.TRUE);
        }

        @Override
        public void cancel() {
        }

        @Override
        public void run() {
        }
    }
}
