package ru.yandex.solomon.locks;

import java.time.Clock;
import java.time.temporal.ChronoUnit;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DistributedLockStub implements DistributedLock {
    private final Clock clock;
    @Nullable
    private volatile LockDetail detail;
    private final AtomicReference<LockSubscriber> subscription = new AtomicReference<>();

    public DistributedLockStub(Clock clock) {
        this.clock = clock;
    }

    public synchronized long setOwner(@Nullable String owner) {
        var prevOwner = this.detail;
        if (prevOwner != null && prevOwner.owner().equals(owner)) {
            return prevOwner.seqNo();
        }

        if (owner == null) {
            this.detail = null;
            if (isLockedByMe(prevOwner)) {
                cancelPrevSubscription();
            }
            return 0;
        }

        LockDetail detail = new LockDetail("", owner, System.nanoTime(), clock.instant().plus(1, ChronoUnit.HOURS));
        this.detail = detail;
        if (isLockedByMe(prevOwner)) {
            cancelPrevSubscription();
        } else {
            notifySubscription();
        }

        return detail.seqNo();
    }

    private void cancelPrevSubscription() {
        var prevSubscription = subscription.getAndSet(null);
        if (prevSubscription != null) {
            prevSubscription.onUnlock(UnlockReason.LEASE_EXPIRED);
        }
    }

    private synchronized void notifySubscription() {
        var detail = this.detail;
        if (!isLockedByMe(detail)) {
            return;
        }

        var subscription = this.subscription.get();
        if (subscription == null) {
            return;
        }

        subscription.onLock(detail.seqNo());
    }

    @Override
    public boolean isLockedByMe() {
        return isLockedByMe(detail);
    }

    private boolean isLockedByMe(@Nullable LockDetail detail) {
        if (detail == null) {
            return false;
        }

        return HostUtils.getFqdn().equals(detail.owner());
    }

    @Override
    public Optional<LockDetail> lockDetail() {
        return Optional.ofNullable(detail);
    }

    @Override
    public CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo) {
        return CompletableFuture.supplyAsync(this::lockDetail);
    }

    @Override
    public void acquireLock(LockSubscriber subscription, long leaseTime, TimeUnit unit) {
        if (subscription.isCanceled()) {
            throw new IllegalArgumentException("LockSubscriber already canceled: " + subscription);
        }
        if (!this.subscription.compareAndSet(null, subscription)) {
            throw new IllegalStateException("unable subscribe on lock twice");
        }
        notifySubscription();
    }

    @Override
    public CompletableFuture<Boolean> unlock() {
        return CompletableFuture.supplyAsync(() -> {
            if (isLockedByMe()) {
                setOwner(null);
                return true;
            }

            return false;
        });
    }
}
