package ru.yandex.solomon.locks;

import java.time.Clock;
import java.time.temporal.ChronoUnit;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ReadOnlyDistributedLockStub implements DistributedLock {
    private final Clock clock;
    @Nullable
    private volatile LockDetail detail;

    public ReadOnlyDistributedLockStub(Clock clock) {
        this.clock = clock;
    }

    public long setOwner(@Nullable String owner) {
        if (owner == null) {
            this.detail = null;
            return 0;
        } else {
            LockDetail detail = new LockDetail("", owner, System.nanoTime(), clock.instant().plus(1, ChronoUnit.HOURS));
            this.detail = detail;
            return detail.seqNo();
        }
    }

    @Override
    public boolean isLockedByMe() {
        LockDetail detail = this.detail;
        if (detail == null) {
            return false;
        }

        return HostUtils.getFqdn().equals(detail.owner());
    }

    @Override
    public Optional<LockDetail> lockDetail() {
        return Optional.ofNullable(detail);
    }

    @Override
    public CompletableFuture<Optional<LockDetail>> getLockDetail(long latestVisibleSeqNo) {
        return CompletableFuture.supplyAsync(this::lockDetail);
    }

    @Override
    public void acquireLock(LockSubscriber subscription, long leaseTime, TimeUnit unit) {
        throw new IllegalStateException("Heartbeat process can't acquire lock");
    }

    @Override
    public CompletableFuture<Boolean> unlock() {
        throw new IllegalStateException("Heartbeat process can't release lock");
    }
}
