package ru.yandex.solomon.locks.dao.memory;

import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.locks.dao.LocksDao;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class InMemoryLocksDao implements LocksDao {
    private final ConcurrentMap<String, LockDetail> locks = new ConcurrentHashMap<>();
    private final Clock clock;
    public volatile Supplier<CompletableFuture<?>> beforeSupplier;

    public InMemoryLocksDao(Clock clock) {
        this.clock = clock;
    }

    @Override
    public CompletableFuture<LockDetail> acquireLock(String lockId, String owner, Instant expiredAt) {
        return before().thenApplyAsync((o) -> {
            return locks.compute(lockId, (k, v) -> {
                if (v == null) {
                    return new LockDetail(lockId, owner, System.nanoTime(), expiredAt);
                }

                if (!v.isExpired(clock.instant())) {
                    return v;
                }

                return new LockDetail(lockId, owner, System.nanoTime(), expiredAt);
            });
        });
    }

    @Override
    public CompletableFuture<Optional<LockDetail>> readLock(String lockId) {
        return before().thenApplyAsync((o) -> {
            return Optional.ofNullable(locks.get(lockId))
                    .filter(lock -> !lock.isExpired(clock.instant()));
        });
    }

    @Override
    public CompletableFuture<Boolean> extendLockTime(String lockId, String owner, Instant expiredAt) {
        return before().thenApplyAsync((o) -> {
            var lock = locks.get(lockId);
            if (isNotOwn(owner, lock)) {
                return Boolean.FALSE;
            }

            return locks.replace(lockId, lock, lock.expiredAt(expiredAt));
        });
    }

    private boolean isNotOwn(String owner, @Nullable LockDetail lock) {
        return lock == null
                || !Objects.equals(lock.owner(), owner)
                || lock.isExpired(clock.instant());
    }

    @Override
    public CompletableFuture<Boolean> releaseLock(String lockId, String owner) {
        return before().thenApplyAsync((o) -> {
            LockDetail lock;
            do {
                lock = locks.get(lockId);
                if (isNotOwn(owner, lock)) {
                    return Boolean.FALSE;
                }
            } while (!locks.remove(lockId, lock));

            return Boolean.TRUE;
        });
    }

    @Override
    public CompletableFuture<List<LockDetail>> listLocks() {
        return before().thenApplyAsync((o) -> List.copyOf(locks.values()));
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return CompletableFuture.completedFuture(null);
        }

        return copy.get();
    }
}
