package ru.yandex.solomon.locks.dao.ydb;

import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AlterTableSettings;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.settings.TtlSettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.PrimitiveType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.locks.dao.LocksDao;

import static com.google.common.base.Preconditions.checkState;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class YdbLocksDao implements LocksDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbLocksDao.class);

    private final SessionRetryContext retryCtx;
    private final Clock clock;
    private final String tablePath;

    private final YdbLocksQuery queries;

    public YdbLocksDao(TableClient tableClient, Clock clock, String rootPath) {
        this.retryCtx = SessionRetryContext.create(tableClient)
            .maxRetries(3)
            .build();
        this.clock = clock;
        this.tablePath = rootPath + "/Locks2";
        this.queries = new YdbLocksQuery(tablePath);
        createSchema();
    }

    public CompletableFuture<Void> createSchema() {
        return retryCtx.supplyResult(session -> session.describeTable(tablePath))
                .thenCompose(result -> {
                    if (!result.isSuccess()) {
                        return createTable();
                    } else {
                        return alterTable(result.expect("unable describe " + tablePath));
                    }
                });
    }

    private CompletableFuture<Void> createTable() {
        TableDescription tableDesc = TableDescription.newBuilder()
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("owner", PrimitiveType.utf8())
                .addNullableColumn("seq_no", PrimitiveType.int64())
                .addNullableColumn("expiration_time", PrimitiveType.int64())
                .addNullableColumn("expire_at", PrimitiveType.timestamp())
                .setPrimaryKey("id")
                .build();

        var settings = new CreateTableSettings();
        settings.setTtlSettings(new TtlSettings("expire_at", (int) TimeUnit.HOURS.toSeconds(1L)));

        return retryCtx.supplyStatus(session -> session.createTable(tablePath, tableDesc, settings))
                .thenAccept(status -> status.expect("cannot create table " + tablePath));
    }

    private CompletableFuture<Void> alterTable(TableDescription table) {
        for (var column : table.getColumns()) {
            if ("expire_at".equals(column.getName())) {
                return completedFuture(null);
            }
        }

        var settings = new AlterTableSettings();
        settings.addColumn("expire_at", PrimitiveType.timestamp().makeOptional());
        settings.setTtlSettings(new TtlSettings("expire_at", (int) TimeUnit.HOURS.toSeconds(1L)));
        return retryCtx.supplyStatus(session -> session.alterTable(tablePath, settings))
                .thenAccept(status -> status.expect("unable alter lock table"));
    }

    @Override
    public CompletableFuture<LockDetail> acquireLock(String lockId, String owner, Instant expiredAt) {
        Params params = Params.of(
            "$id", utf8(lockId),
            "$owner", utf8(owner),
            "$expiration_time", int64(expiredAt.toEpochMilli()),
            "$expire_at", timestamp(expiredAt),
            "$now", timestamp(clock.instant()),
            "$seq_no", int64(clock.millis()));

        return executeQuery(queries.acquire, params)
            .thenApply(result -> {
                DataQueryResult dataQueryResult = result.expect("cannot acquire lock " + lockId);
                ResultSetReader resultSet = dataQueryResult.getResultSet(0);
                checkState(resultSet.next(), "empty result set");
                return toLockDetail(resultSet);
            });
    }

    @Override
    public CompletableFuture<Optional<LockDetail>> readLock(String lockId) {
        Params params = Params.of(
            "$id", utf8(lockId),
            "$now", timestamp(clock.instant()));

        return executeQuery(queries.read, params)
            .thenApply(result -> {
                DataQueryResult dataQueryResult = result.expect("cannot read lock " + lockId);
                ResultSetReader resultSet = dataQueryResult.getResultSet(0);
                if (!resultSet.next()) {
                    return Optional.empty();
                }
                return Optional.of(toLockDetail(resultSet));
            });
    }

    @Override
    public CompletableFuture<Boolean> extendLockTime(String lockId, String owner, Instant expiredAt) {
        Params params = Params.of(
            "$id", utf8(lockId),
            "$owner", utf8(owner),
            "$expiration_time", int64(expiredAt.toEpochMilli()),
            "$expire_at", timestamp(expiredAt),
            "$now", timestamp(clock.instant()));

        return executeQuery(queries.extend, params)
            .thenApply(result -> {
                DataQueryResult dataQueryResult = result.expect("cannot extend lock " + lockId);
                ResultSetReader resultSet = dataQueryResult.getResultSet(0);
                checkState(resultSet.next(), "empty result set");
                return resultSet.getColumn(0).getUint32() != 0;
            });
    }

    @Override
    public CompletableFuture<Boolean> releaseLock(String lockId, String owner) {
        Params params = Params.of(
            "$id", utf8(lockId),
            "$owner", utf8(owner),
            "$now", timestamp(clock.instant()));

        return executeQuery(queries.release, params)
            .thenApply(result -> {
                DataQueryResult dataQueryResult = result.expect("cannot release lock " + lockId);
                ResultSetReader resultSet = dataQueryResult.getResultSet(0);
                checkState(resultSet.next(), "empty result set");
                return resultSet.getColumn(0).getUint32() != 0;
            });
    }

    @Override
    public CompletableFuture<List<LockDetail>> listLocks() {
        return listLocksBySelect()
                .handle((result, e) -> {
                    if (e != null) {
                        logger.warn("Unable find resource by select, fallback to readtable", e);
                        return listLocksByReadTable();
                    }

                    if (result == null) {
                        // more records that limit for select
                        return listLocksByReadTable();
                    }

                    return completedFuture(result);
                })
                .thenCompose(future -> future);
    }

    @Nullable
    private CompletableFuture<List<LockDetail>> listLocksBySelect() {
        return executeQuery(queries.list, Params.empty())
                .thenApply(result -> {
                    DataQueryResult dataQueryResult = result.expect("cannot select locks");
                    ResultSetReader resultSet = dataQueryResult.getResultSet(0);
                    if (resultSet.isTruncated() || resultSet.getRowCount() >= 1000) {
                        return null;
                    }

                    var reader = new RecordReader();
                    reader.accept(resultSet);
                    return reader.records;
                });
    }

    private CompletableFuture<List<LockDetail>> listLocksByReadTable() {
        var reader = new RecordReader();
        return retryCtx.supplyStatus(session -> {
            var settings = ReadTableSettings.newBuilder()
                    .timeout(1, TimeUnit.MINUTES)
                    .orderedRead(true);

            if (reader.last != null) {
                settings.fromKeyExclusive(utf8(reader.last.id()));
            }

            return session.readTable(tablePath, settings.build(), reader);
        }).thenApply(status -> {
            status.expect("can't read " + tablePath);
            return reader.records;
        });
    }

    private CompletableFuture<Result<DataQueryResult>> executeQuery(String query, Params params) {
        var txControl = TxControl.serializableRw().setCommitTx(true);
        var setting = new ExecuteDataQuerySettings().keepInQueryCache();
        return retryCtx.supplyResult(session -> session.executeDataQuery(query, txControl, params, setting));
    }

    private static LockDetail toLockDetail(ResultSetReader r) {
        var reader = new ColumnReader(r);
        return reader.read(r);
    }
}
