package ru.yandex.solomon.locks;

import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.memory.InMemoryLocksDao;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class DistributedLockImplTest {
    private static final String LOCK_ID = "my_lock";

    @Rule
    public Timeout globalTimeout = Timeout.builder()
            .withTimeout(3, TimeUnit.SECONDS)
            .withLookingForStuckThread(true)
            .build();

    private ManualClock clock;
    private ScheduledExecutorService executor;
    private LocksDao dao;

    @Before
    public void setUp() throws Exception {
        this.clock = new ManualClock();
        this.executor = new ManualScheduledExecutorService(4, clock);
        this.dao = new InMemoryLocksDao(clock);
    }

    @After
    public void tearDown() throws Exception {
        this.executor.shutdownNow();
    }

    @Test
    public void absentLockByDefault() throws InterruptedException {
        DistributedLock lock = app("Alice");
        assertThat(lock.isLockedByMe(), equalTo(false));
        assertThat(lock.lockDetail().map(LockDetail::owner), equalTo(Optional.empty()));

        clock.passedTime(1, TimeUnit.MINUTES);
        TimeUnit.MILLISECONDS.sleep(10L);

        assertThat(lock.isLockedByMe(), equalTo(false));
        assertThat(lock.lockDetail().map(LockDetail::owner), equalTo(Optional.empty()));
    }

    @Test
    public void acquireLock() throws InterruptedException {
        DistributedLock alice = app("Alice");

        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        assertThat(alice.isLockedByMe(), equalTo(true));
        assertThat(alice.lockDetail().map(LockDetail::owner).orElse(null), equalTo("Alice"));
    }

    @Test
    public void extendAcquiredLock() throws InterruptedException {
        DistributedLock alice = app("Alice");

        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        for (int index = 0; index < 30; index++) {
            clock.passedTime(1, TimeUnit.SECONDS);
            TimeUnit.MILLISECONDS.sleep(1L);
        }

        assertThat(alice.isLockedByMe(), equalTo(true));
        assertThat(alice.lockDetail().map(LockDetail::owner).orElse(null), equalTo("Alice"));
    }

    @Test
    public void unlockByExpireReason() throws InterruptedException {
        DistributedLock alice = app("Alice");
        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        for (int index = 1; index <= 30; index++) {
            clock.passedTime(index * index, TimeUnit.SECONDS);
            TimeUnit.MILLISECONDS.sleep(1L);
        }

        aliceSub.syncUnlock.await();
        assertThat(alice.isLockedByMe(), equalTo(false));
        assertThat(alice.lockDetail().map(LockDetail::owner), equalTo(Optional.empty()));
        assertThat(aliceSub.unlockReason, equalTo(UnlockReason.LEASE_EXPIRED));
    }

    @Test
    public void unlockManual() throws InterruptedException {
        DistributedLock alice = app("Alice");
        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 60, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        Boolean result = alice.unlock().join();
        assertThat(result, equalTo(true));

        aliceSub.syncUnlock.await();
        assertThat(alice.isLockedByMe(), equalTo(false));
        assertThat(alice.lockDetail().map(LockDetail::owner), equalTo(Optional.empty()));
        assertThat(aliceSub.unlockReason, equalTo(UnlockReason.MANUAL_UNLOCK));
    }

    @Test
    public void onlyOneOwner() throws InterruptedException {
        DistributedLock alice = app("Alice");
        DistributedLock bob = app("Bob");

        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 60, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        SyncSubscription bobSub = new SyncSubscription();
        bob.acquireLock(bobSub, 60, TimeUnit.SECONDS);
        clock.passedTime(10, TimeUnit.SECONDS);
        bobSub.awaitLock(30, TimeUnit.MILLISECONDS);

        assertThat(alice.isLockedByMe(), equalTo(true));
        assertThat(alice.lockDetail().map(LockDetail::owner).orElse(null), equalTo("Alice"));

        assertThat(bob.isLockedByMe(), equalTo(false));
        assertThat(bob.lockDetail().map(LockDetail::owner).orElse("Alice"), equalTo("Alice"));
    }

    @Test
    public void acquireLockWhenOwnershipExpired() throws InterruptedException {
        DistributedLock alice = app("Alice");
        DistributedLock bob = app("Bob");

        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        clock.passedTime(30, TimeUnit.SECONDS);
        aliceSub.syncUnlock.await();
        assertThat(aliceSub.unlockReason, equalTo(UnlockReason.LEASE_EXPIRED));

        SyncSubscription bobSub = new SyncSubscription();
        bob.acquireLock(bobSub, 30, TimeUnit.SECONDS);
        bobSub.awaitLock();
        assertThat(bob.isLockedByMe(), equalTo(true));
        assertThat(bob.lockDetail().map(LockDetail::owner).orElse(null), equalTo("Bob"));

        while (alice.lockDetail().map(LockDetail::owner).isEmpty()) {
            clock.passedTime(1, TimeUnit.SECONDS);
            TimeUnit.MILLISECONDS.sleep(5);
        }

        assertThat(alice.isLockedByMe(), equalTo(false));
        assertThat(alice.lockDetail().map(LockDetail::owner).orElse(null), equalTo("Bob"));
    }

    @Test
    public void acquireLockWhenPrevOwnerReleaseIt() throws InterruptedException {
        DistributedLock alice = app("Alice");
        DistributedLock bob = app("Bob");

        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        SyncSubscription bobSub = new SyncSubscription();
        bob.acquireLock(bobSub, 30, TimeUnit.SECONDS);
        bobSub.awaitLock(ThreadLocalRandom.current().nextLong(0, 10), TimeUnit.MILLISECONDS);

        assertThat(alice.unlock().join(), equalTo(true));
        clock.passedTime(30, TimeUnit.SECONDS);
        do {
            clock.passedTime(2, TimeUnit.SECONDS);
        } while (!bobSub.syncLock.await(2, TimeUnit.MILLISECONDS));

        assertThat(bob.isLockedByMe(), equalTo(true));
        assertThat(bob.lockDetail().map(LockDetail::owner).orElse(null), equalTo("Bob"));

        assertThat(alice.isLockedByMe(), equalTo(false));
        assertThat(alice.lockDetail().map(LockDetail::owner).orElse("Bob"), equalTo("Bob"));
    }

    @Test(expected = IllegalArgumentException.class)
    public void notAbleSubscribeTwice() throws InterruptedException {
        DistributedLock alice = app("Alice");

        SyncSubscription subOne = new SyncSubscription();
        SyncSubscription subTwo = new SyncSubscription();

        alice.acquireLock(subOne, 30, TimeUnit.SECONDS);
        TimeUnit.MILLISECONDS.sleep(ThreadLocalRandom.current().nextLong(0, 30));
        alice.acquireLock(subTwo, 30, TimeUnit.SECONDS);
    }

    @Test
    public void releaseLockOnPersistenceError() throws InterruptedException {
        dao = new InMemoryLocksDao(clock) {
            @Override
            public CompletableFuture<Boolean> extendLockTime(String lockId, String owner, Instant expiredAt) {
                return CompletableFuture.failedFuture(new IllegalStateException("Hi"));
            }
        };

        DistributedLock alice = app("Alice");

        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        clock.passedTime(15, TimeUnit.SECONDS);
        while (!aliceSub.syncUnlock.await(2, TimeUnit.MILLISECONDS)) {
            clock.passedTime(2, TimeUnit.SECONDS);
        }
        assertThat(aliceSub.unlockReason, equalTo(UnlockReason.CONNECTION_LOST));
    }

    @Test
    public void retryExtendLockWhenError() throws InterruptedException {
        AtomicInteger index = new AtomicInteger();
        AtomicReference<CountDownLatch> sync = new AtomicReference<>(new CountDownLatch(1));
        dao = new InMemoryLocksDao(clock) {
            @Override
            public CompletableFuture<Boolean> extendLockTime(String lockId, String owner, Instant expiredAt) {
                CountDownLatch copy = sync.get();
                try {
                    var i = index.incrementAndGet();
                    if (i % 2 == 0 || i > 5) {
                        return failedFuture(new IllegalStateException("Predefine error"));
                    }

                    return completedFuture(Boolean.TRUE);
                } finally {
                    sync.compareAndSet(copy, new CountDownLatch(1));
                    copy.countDown();
                }
            }
        };

        DistributedLock alice = app("Alice");

        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);
        aliceSub.awaitLock();

        do {
            var s = sync.get();
            do {
                clock.passedTime(1, TimeUnit.SECONDS);
            } while (!s.await(2, TimeUnit.MILLISECONDS));
        } while (index.get() < 5);

        assertFalse(aliceSub.syncUnlock.await(10, TimeUnit.MILLISECONDS));

        clock.passedTime(30, TimeUnit.SECONDS);

        assertTrue(aliceSub.syncUnlock.await(100, TimeUnit.MILLISECONDS));
        assertEquals(UnlockReason.CONNECTION_LOST, aliceSub.unlockReason);
    }

    @Test
    public void errorOnAcquireDoNotStopAcquire() throws InterruptedException {
        AtomicInteger failCount = new AtomicInteger();
        dao = new InMemoryLocksDao(clock) {
            @Override
            public CompletableFuture<LockDetail> acquireLock(String lockId, String owner, Instant expiredAt) {
                if (failCount.incrementAndGet() > 5) {
                    return super.acquireLock(lockId, owner, expiredAt);
                } else {
                    return CompletableFuture.failedFuture(new IllegalStateException("Hi"));
                }
            }
        };

        DistributedLock alice = app("Alice");
        SyncSubscription aliceSub = new SyncSubscription();
        alice.acquireLock(aliceSub, 30, TimeUnit.SECONDS);

        do {
            clock.passedTime(1, TimeUnit.SECONDS);
        } while (!aliceSub.awaitLock(5, TimeUnit.MILLISECONDS));

        assertThat(alice.isLockedByMe(), equalTo(true));
        assertThat(alice.lockDetail().map(LockDetail::owner).orElse(null), equalTo("Alice"));
    }

    private DistributedLockImpl app(String appId) {
        return new DistributedLockImpl(LOCK_ID, LOCK_ID, appId, dao, clock, executor, new MetricRegistry());
    }

    private class SyncSubscription implements LockSubscriber {
        CountDownLatch syncLock = new CountDownLatch(1);
        CountDownLatch syncUnlock = new CountDownLatch(1);

        @Nullable
        volatile UnlockReason unlockReason;

        @Override
        public boolean isCanceled() {
            return false;
        }

        @Override
        public void onLock(long seqNo) {
            syncLock.countDown();
        }

        @Override
        public void onUnlock(UnlockReason reason) {
            unlockReason = reason;
            syncUnlock.countDown();
        }

        private void awaitLock() throws InterruptedException {
            clock.passedTime(2, TimeUnit.SECONDS);
            syncLock.await();
        }

        private boolean awaitLock(long timeout, TimeUnit unit) throws InterruptedException {
            clock.passedTime(2, TimeUnit.SECONDS);
            return syncLock.await(timeout, unit);
        }
    }
}
