package ru.yandex.solomon.locks.dao;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.IntStream;

import org.junit.After;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.util.async.InFlightLimiter;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class LocksDaoTest {
    public static final ManualClock CLOCK = new ManualClock();

    @Rule
    public TestName name = new TestName();

    @After
    public void tearDown() throws Exception {
        CLOCK.passedTime(1, TimeUnit.HOURS);
    }

    public abstract LocksDao getDao();

    @Test
    public void readNotExistLock() {
        Optional<LockDetail> opt = getDao().readLock("not_exists").join();
        assertThat(opt, equalTo(Optional.empty()));
    }

    @Test
    public void acquireFreeLock() {
        Instant expiredAt = CLOCK.instant().plus(10, ChronoUnit.MINUTES);
        LockDetail result = acquireLock("freeLock", name.getMethodName(), expiredAt);

        assertThat(result, notNullValue());
        assertThat(result.owner(), equalTo(name.getMethodName()));
        assertThat(result.expiredAt(), equalTo(expiredAt));
    }

    @Test
    public void readAcquiredLock() {
        Instant expiredAt = CLOCK.instant().plus(10, ChronoUnit.MINUTES);
        LockDetail locked = acquireLock("forReadLock", name.getMethodName(), expiredAt);
        Optional<LockDetail> read = getDao().readLock("forReadLock").join();

        assertThat(read.map(LockDetail::owner).orElse(null), equalTo(locked.owner()));
        assertThat(read.map(LockDetail::expiredAt).orElse(null), equalTo(locked.expiredAt()));
    }

    @Test
    public void readExpiredLock() {
        LockDetail result = acquireLock("forReadExpired", name.getMethodName(), CLOCK.instant().plusSeconds(40));
        assertThat(result.owner(), equalTo(name.getMethodName()));

        CLOCK.passedTime(60, TimeUnit.SECONDS);
        Optional<LockDetail> read = getDao().readLock("forReadExpired").join();
        assertThat(read, equalTo(Optional.empty()));
    }

    @Test
    public void notAbleAcquireLockUntilUnlockOrExpire() {
        final String ownerOne = name.getMethodName() + "_1";
        final String ownerTwo = name.getMethodName() + "_2";

        LockDetail one = acquireLock("twiceAcquire", ownerOne, CLOCK.instant().plusSeconds(30));
        LockDetail two = acquireLock("twiceAcquire", ownerTwo, CLOCK.instant().plusSeconds(40));

        assertThat(one.owner(), equalTo(ownerOne));
        assertThat(two.owner(), equalTo(ownerOne));
    }

    @Test
    public void acquireExpiredLock() {
        final String ownerOne = name.getMethodName() + "_1";
        final String ownerTwo = name.getMethodName() + "_2";

        LockDetail one = acquireLock("expireTest", ownerOne, CLOCK.instant().plusSeconds(60));
        assertThat(one.owner(), equalTo(ownerOne));
        CLOCK.passedTime(60, TimeUnit.SECONDS);

        Instant expiredAt = CLOCK.instant().plusSeconds(60);
        LockDetail two = acquireLock("expireTest", ownerTwo, expiredAt);
        assertThat(two.owner(), equalTo(ownerTwo));
        assertThat(two.expiredAt(), equalTo(expiredAt));

        LockDetail read = getDao().readLock("expireTest").join().get();
        assertThat(read.owner(), equalTo(ownerTwo));
        assertThat(read.expiredAt(), equalTo(expiredAt));
    }

    @Test
    public void extendNotExistLock() {
        Boolean result = getDao()
            .extendLockTime("not_exist", name.getMethodName(), CLOCK.instant().plusSeconds(60))
            .join();

        assertThat(result, equalTo(Boolean.FALSE));
    }

    @Test
    public void extendNotAcquiredLock() {
        final String ownerOne = name.getMethodName() + "_1";
        final String ownerTwo = name.getMethodName() + "_2";

        LockDetail one = acquireLock("extendNotOwn", ownerOne, CLOCK.instant().plusSeconds(60));
        assertThat(one.owner(), equalTo(ownerOne));

        Boolean result = getDao()
            .extendLockTime("extendNotOwn", ownerTwo, CLOCK.instant().plusSeconds(80))
            .join();

        assertThat(result, equalTo(Boolean.FALSE));
    }

    @Test
    public void extendExpiredLock() {
        LockDetail one = acquireLock("extendExpired", name.getMethodName(), CLOCK.instant().plusSeconds(60));
        assertThat(one.owner(), equalTo(name.getMethodName()));
        CLOCK.passedTime(60, TimeUnit.SECONDS);

        Boolean result = getDao()
            .extendLockTime("extendExpired", name.getMethodName(), CLOCK.instant().plusSeconds(60))
            .join();

        assertThat(result, equalTo(Boolean.FALSE));
    }

    @Test
    public void extendLock() {
        Instant expiredAt = CLOCK.instant().plusSeconds(30);
        LockDetail locked = acquireLock("forExtend", name.getMethodName(), expiredAt);
        assertThat(locked.owner(), equalTo(name.getMethodName()));

        for (int index = 0; index < 30; index++) {
            CLOCK.passedTime(5, TimeUnit.SECONDS);
            expiredAt = CLOCK.instant().plusSeconds(30);
            Boolean result = getDao()
                .extendLockTime("forExtend", name.getMethodName(), expiredAt)
                .join();

            assertThat(result, equalTo(Boolean.TRUE));
        }

        LockDetail read = getDao().readLock("forExtend").join().get();
        assertThat(read.owner(), equalTo(name.getMethodName()));
        assertThat(read.expiredAt(), equalTo(expiredAt));
    }

    @Test
    public void unlock() {
        LockDetail locked = acquireLock("forUnlock", name.getMethodName(), CLOCK.instant().plusSeconds(40));
        assertThat(locked.owner(), equalTo(name.getMethodName()));

        Boolean result = getDao().releaseLock("forUnlock", name.getMethodName()).join();
        assertThat(result, equalTo(Boolean.TRUE));

        Optional<LockDetail> read = getDao().readLock("forUnlock").join();
        assertThat(read, equalTo(Optional.empty()));
    }

    @Test
    public void unlockExpired() {
        LockDetail locked = acquireLock("forUnlockExpire", name.getMethodName(), CLOCK.instant().plusSeconds(50));
        assertThat(locked.owner(), equalTo(name.getMethodName()));

        CLOCK.passedTime(60, TimeUnit.SECONDS);
        Boolean result = getDao().releaseLock("forUnlockExpire", name.getMethodName()).join();
        assertThat(result, equalTo(Boolean.FALSE));
    }

    @Test
    public void unlockNotOwned() {
        final String ownerOne = name.getMethodName() + "_1";
        final String ownerTwo = name.getMethodName() + "_2";

        Instant expiredAt = CLOCK.instant().plusSeconds(50);
        LockDetail locked = acquireLock("forUnlockNotOwned", ownerOne, expiredAt);
        assertThat(locked.owner(), equalTo(ownerOne));

        Boolean result = getDao().releaseLock("forUnlockNotOwned", ownerTwo).join();
        assertThat(result, equalTo(Boolean.FALSE));

        Optional<LockDetail> read = getDao().readLock("forUnlockNotOwned").join();
        assertThat(read.map(LockDetail::owner).orElse(null), equalTo(ownerOne));
        assertThat(read.map(LockDetail::expiredAt).orElse(null), equalTo(expiredAt));
    }

    @Test
    public void unlockNotExist() {
       Boolean result = getDao().releaseLock("not_exist", name.getMethodName()).join();
       assertThat(result, equalTo(Boolean.FALSE));
    }

    @Test
    public void lockUnlockSeqNoGrow() {
        LockDetail lockOne = acquireLock("myLock", name.getMethodName(), CLOCK.instant().plusSeconds(40));
        assertThat(lockOne.owner(), equalTo(name.getMethodName()));

        Boolean result = getDao().releaseLock("myLock", name.getMethodName()).join();
        assertThat(result, equalTo(Boolean.TRUE));

        LockDetail lockTwo = acquireLock("myLock", name.getMethodName(), CLOCK.instant().plusSeconds(40));
        assertThat(lockTwo.owner(), equalTo(name.getMethodName()));
        assertThat(lockTwo.seqNo(), greaterThan(lockOne.seqNo()));
    }

    @Test
    public void lockExpireLockSeqNoGrow() {
        LockDetail lockOne = acquireLock("leader", name.getMethodName(), CLOCK.instant().plusSeconds(40));
        assertThat(lockOne.owner(), equalTo(name.getMethodName()));

        CLOCK.passedTime(2, TimeUnit.MINUTES);
        LockDetail lockTwo = acquireLock("leader", name.getMethodName(), CLOCK.instant().plusSeconds(60));
        assertThat(lockTwo.owner(), equalTo(name.getMethodName()));
        assertThat(lockTwo.seqNo(), greaterThan(lockOne.seqNo()));
    }

    @Test
    public void acquireLockReplaceObsoleteRace() throws ExecutionException, InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(2, r -> {
            Thread thread = new Thread(r);
            thread.setName(name.getMethodName());
            thread.setDaemon(true);
            return thread;
        });

        CyclicBarrier barrier = new CyclicBarrier(2, () -> {
            CLOCK.passedTime(1, TimeUnit.MINUTES);

            // create expired lock to replace with new one
            LockDetail lockOne = acquireLock(name.getMethodName(), "obsoleteThread", CLOCK.instant().plusSeconds(30));
            assertThat(lockOne.owner(), equalTo("obsoleteThread"));
            CLOCK.passedTime(40, TimeUnit.SECONDS);
        });

        Callable<LockDetail> action = () -> {
            LocksDao dao = getDao();
            String lockId = name.getMethodName();
            String owner = Thread.currentThread().getName();
            Instant expiredAt = CLOCK.instant().plusSeconds(360);
            barrier.await();
            return dao.acquireLock(lockId, owner, expiredAt).join();
        };

        Future<LockDetail> f1 = executor.submit(action);
        Future<LockDetail> f2 = executor.submit(action);

        LockDetail lockOne = f1.get();
        LockDetail lockTwo = f2.get();
        executor.shutdownNow();

        assertThat("Both of the thread after try acquire lock should see the same state",
                lockOne, equalTo(lockTwo));
    }

    @Test
    public void acquireLockInsertRace() throws ExecutionException, InterruptedException {
        ExecutorService executor = Executors.newFixedThreadPool(2, r -> {
            Thread thread = new Thread(r);
            thread.setName(name.getMethodName());
            thread.setDaemon(true);
            return thread;
        });

        CyclicBarrier barrier = new CyclicBarrier(2);

        Callable<LockDetail> action = () -> {
            LocksDao dao = getDao();
            String lockId = name.getMethodName();
            String owner = Thread.currentThread().getName();
            Instant expiredAt = CLOCK.instant().plusSeconds(60);
            barrier.await();
            return dao.acquireLock(lockId, owner, expiredAt).join();
        };

        Future<LockDetail> f1 = executor.submit(action);
        Future<LockDetail> f2 = executor.submit(action);

        LockDetail lockOne = f1.get();
        LockDetail lockTwo = f2.get();
        executor.shutdownNow();

        assertThat("Both of the thread after try acquire lock should see the same state",
                lockOne, equalTo(lockTwo));
    }

    @Test
    public void listLocksEmpty() {
        var result = getDao().listLocks().join();
        assertEquals(List.of(), result);
    }

    @Test
    public void listLocks() {
        var alice = acquireLock("alice", "alice-owner", CLOCK.instant().plusSeconds(30));
        var bob = acquireLock("bob", "bob-owner", CLOCK.instant().plusSeconds(35));

        var list = getDao().listLocks().join();
        assertEquals(Set.of(alice, bob), Set.copyOf(list));
    }

    @Test
    public void listManyLocks() {
        InFlightLimiter limiter = new InFlightLimiter(10);
        var dao = getDao();
        var acquired = IntStream.range(0, 1005)
                .parallel()
                .mapToObj(id -> {
                    CompletableFuture<LockDetail> doneFuture = new CompletableFuture<>();
                    limiter.run(() -> {
                        var future = dao.acquireLock(
                                "lockId-" + id, "alice", CLOCK.instant().plus(1, ChronoUnit.HOURS));
                        CompletableFutures.whenComplete(future, doneFuture);
                        return future;
                    });
                    return doneFuture;
                })
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(Set::copyOf)
                .join();

        var list = getDao().listLocks().join();
        assertEquals(acquired, Set.copyOf(list));
    }

    @Test
    public void listLocksNotIncludeUnlocked() {
        var alice = acquireLock("alice", name.getMethodName(), CLOCK.instant().plusSeconds(30));
        assertEquals(List.of(alice), getDao().listLocks().join());
        assertTrue(getDao().releaseLock("alice", name.getMethodName()).join());
        assertEquals(List.of(), getDao().listLocks().join());
    }

    private LockDetail acquireLock(String lockId, String owner, Instant expiredAt) {
        return getDao().acquireLock(lockId, owner, expiredAt).join();
    }
}
