package ru.yandex.solomon.scheduler;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.Nullable;

import com.google.protobuf.Any;
import com.google.protobuf.Message;
import io.grpc.Status;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.Proto;

/**
 * @author Vladimir Gordiychuk
 */
public class ExecutionContextStub implements ExecutionContext {
    private final Task task;

    public final ArrayBlockingQueue<Event> events = new ArrayBlockingQueue<>(10000);
    public volatile boolean autoComplete = true;
    private final AtomicBoolean done = new AtomicBoolean();
    private final CompletableFuture<Event> doneFuture = new CompletableFuture<>();

    public ExecutionContextStub(Task task) {
        this.task = task;
    }

    @Override
    public Task task() {
        return task;
    }

    @Override
    public boolean isDone() {
        return done.get();
    }

    @Override
    public <T extends Message> CompletableFuture<?> complete(T result) {
        return enqueueDone(new Complete(Proto.pack(result), new CompletableFuture<>()));
    }

    @Override
    public CompletableFuture<?> fail(Throwable e) {
        return enqueueDone(new Fail(e, new CompletableFuture<>()));
    }

    @Override
    public <T extends Message> CompletableFuture<?> reschedule(long executeAt, T progress) {
        return enqueueDone(new Reschedule(executeAt, Proto.pack(progress), new CompletableFuture<>()));
    }

    @Override
    public <T extends Message> CompletableFuture<?> progress(T progress) {
        return enqueueEvent(new Progress(Proto.pack(progress), new CompletableFuture<>()));
    }

    @Override
    public CompletableFuture<?> cancel() {
        return enqueueDone(new Cancel(new CompletableFuture<>()));
    }

    private CompletableFuture<?> enqueueDone(Event event) {
        if (!done.compareAndSet(false, true)) {
            return CompletableFuture.failedFuture(Status.FAILED_PRECONDITION
                    .withDescription("Task already done")
                    .asRuntimeException());
        }

        CompletableFutures.whenComplete(event.future().thenApply(ignore -> event), doneFuture);
        events.add(event);
        if (autoComplete) {
            event.future().complete(null);
        }
        return event.future();
    }

    private CompletableFuture<?> enqueueEvent(Event event) {
        if (done.get()) {
            return CompletableFuture.failedFuture(Status.FAILED_PRECONDITION
                    .withDescription("Task already done")
                    .asRuntimeException());
        }

        events.add(event);
        if (autoComplete) {
            event.future().complete(null);
        }
        return event.future();
    }

    public void markCompleted() {
        done.set(true);
        Event event;
        while ((event = events.poll()) != null) {
            event.future().completeExceptionally(Status.FAILED_PRECONDITION
                    .withDescription("Task already done")
                    .asRuntimeException());
        }
    }

    public <T extends Event> T takeEvent(Class<T> clazz) {
        return takeEvent(clazz, false);
    }

    public <T extends Event> T takeDoneEvent(Class<T> clazz) {
        return takeEvent(clazz, true);
    }

    public <T extends Event> T takeEvent(Class<T> clazz, boolean skipProgress) {
        while (true) {
            Event event = takeEvent();
            if (skipProgress && event instanceof Progress) {
                event.future().complete(null);
                continue;
            }

            if (!clazz.isInstance(event)) {
                throw new ClassCastException("Can not cast " + event + " to " + clazz.getName());
            }

            return clazz.cast(event);
        }
    }

    public Event takeEvent() {
        try {
            return events.take();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    @Nullable
    public Event poolEvent(long time, TimeUnit unit) {
        try {
            return events.poll(time, unit);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }

    @Nullable
    public <T extends Event> T poolEvent(Class<T> clazz, long time, TimeUnit unit) {
        return poolEvent(clazz, false, time, unit);
    }

    @Nullable
    public <T extends Event> T poolEvent(Class<T> clazz, boolean skipProgress, long time, TimeUnit unit) {
        while (true) {
            Event event = poolEvent(time, unit);
            if (skipProgress && event instanceof Progress) {
                event.future().complete(null);
                continue;
            }

            return clazz.cast(event);
        }
    }

    public void expectNoEvents() {
        if (!events.isEmpty()) {
            throw new IllegalStateException("events queue size on context should be zero");
        }
    }

    public <T extends Event> CompletableFuture<T> expectDone(Class<T> clazz) {
        return doneFuture.thenApply(clazz::cast);
    }

    public interface Event {
        CompletableFuture<?> future();
    }

    public record Reschedule(long executeAt, Any progress, CompletableFuture<?> future) implements Event {}
    public record Progress(Any progress, CompletableFuture<?> future) implements Event {}
    public record Cancel(CompletableFuture<?> future) implements Event {}
    public record Fail(Throwable throwable, CompletableFuture<?> future) implements Event {}
    public record Complete(Any result, CompletableFuture<?> future) implements Event {}
}
