package ru.yandex.solomon.scheduler;

import com.google.protobuf.Any;
import io.grpc.Status;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public record Task(
    String id,
    String type,
    Any params,
    long executeAt,
    State state,
    Any progress,
    Status status,
    Any result,
    int version)
{

    public Task {
        requireNonNull(id);
        requireNonNull(type);
        requireNonNull(params);
        requireNonNull(state);
        requireNonNull(progress);
        requireNonNull(status);
        requireNonNull(result);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static class Builder {
        private String id;
        private String type;
        private Any params = Any.getDefaultInstance();
        private long executeAt;
        private State state = State.SCHEDULED;
        private Any progress = Any.getDefaultInstance();
        private Any result = Any.getDefaultInstance();
        private Status status = Status.OK;
        private int version;

        public Builder() {
        }

        public Builder(Task task) {
            this.id = task.id;
            this.type = task.type;
            this.params = task.params;
            this.executeAt = task.executeAt;
            this.state = task.state;
            this.progress = task.progress;
            this.result = task.result;
            this.status = task.status;
            this.version = task.version;
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public Builder setType(String type) {
            this.type = type;
            return this;
        }

        public Builder setParams(Any params) {
            this.params = params;
            return this;
        }

        public Builder setResult(Any result) {
            this.result = result;
            return this;
        }

        public Builder setExecuteAt(long executeAt) {
            this.executeAt = executeAt;
            return this;
        }

        public Builder setState(State state) {
            this.state = state;
            return this;
        }

        public Builder setStatus(Status status) {
            this.status = status;
            return this;
        }

        public Builder setProgress(Any progress) {
            this.progress = progress;
            return this;
        }

        public Builder setVersion(int version) {
            this.version = version;
            return this;
        }

        public Task build() {
            return new Task(id, type, params, executeAt, state, progress, status, result, version);
        }
    }

    public enum State {
        SCHEDULED(0), RUNNING(1), COMPLETED(2), UNKNOWN(-1);

        private static final State[] VALUES = values();
        public final int number;

        State(int number) {
            this.number = number;
        }

        public static State valueOf(int number) {
            for (var state : VALUES) {
                if (state.number == number) {
                    return state;
                }
            }

            return UNKNOWN;
        }
    }
}
