package ru.yandex.solomon.scheduler;

import java.util.EnumMap;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import io.grpc.Status;
import io.grpc.Status.Code;
import it.unimi.dsi.fastutil.objects.Object2LongOpenHashMap;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;

/**
 * @author Vladimir Gordiychuk
 */
public class TaskMetrics implements MetricSupplier {
    private final ConcurrentMap<String, Type> metricsByType = new ConcurrentHashMap<>();

    public Type getByType(String type) {
        var result = metricsByType.get(type);
        if (result != null) {
            return result;
        }

        return metricsByType.computeIfAbsent(type, Type::new);
    }

    public void updateScheduled(List<ScheduledTask> scheduled) {
        if (scheduled.isEmpty()) {
            for (var metrics : metricsByType.values()) {
                metrics.waitToExec.set(0);
            }
            return;
        }

        Object2LongOpenHashMap<String> count = new Object2LongOpenHashMap<>();
        count.defaultReturnValue(0);

        for (var task : scheduled) {
            count.addTo(task.type(), 1);
        }

        for (var metrics : metricsByType.values()) {
            metrics.waitToExec.set(count.getLong(metrics.type));
        }
    }

    @Override
    public int estimateCount() {
        if (metricsByType.isEmpty()) {
            return 0;
        }

        return metricsByType.values().iterator().next().estimateCount() * (metricsByType.size() + 1);
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        var total = new Type("total");
        for (var metrics : metricsByType.values()) {
            total.combine(metrics);
            metrics.append(tsMillis, commonLabels, consumer);
        }

        total.append(tsMillis, commonLabels, consumer);
    }

    public static class Type implements MetricSupplier {
        final String type;

        final Rate schedule;
        final Rate reschedule;
        final Rate rescheduleExternally;
        final Rate progress;
        final Rate cancel;
        final Histogram elapsedMillis;
        final GaugeInt64 waitToExec;
        final Histogram lagMillis;
        final AsyncMetrics pipeline;

        final ConcurrentMap<Code, Rate> status;
        final ConcurrentMap<Code, Rate> pipelineStatus;
        final EnumMap<TaskPipeline.State, Rate> pipelineTimeInState;

        private final MetricRegistry registry;

        public Type(String type) {
            this.type = type;
            this.registry = new MetricRegistry(Labels.of("type", type));
            this.schedule = registry.rate("scheduler.tasks.schedule");
            this.reschedule = registry.rate("scheduler.tasks.reschedule");
            this.rescheduleExternally = registry.rate("scheduler.tasks.reschedule_externally");
            this.progress = registry.rate("scheduler.tasks.progress");
            this.cancel = registry.rate("scheduler.tasks.cancel");
            this.status = new ConcurrentHashMap<>();
            this.elapsedMillis = registry.histogramRate("scheduler.tasks.elapsed_millis", Histograms.exponential(20, 2));
            this.waitToExec = registry.gaugeInt64("scheduler.tasks.wait_count");
            this.lagMillis = registry.histogramRate("scheduler.tasks.lag_millis", Histograms.exponential(20, 2));
            this.pipeline = new AsyncMetrics(registry, "scheduler.tasks.pipeline.");
            this.pipelineStatus = new ConcurrentHashMap<>();
            this.pipelineTimeInState = EnumMetrics.rates(TaskPipeline.State.class, registry,
                    "scheduler.task.pipeline.time_in_state_nanos", "state");
        }

        @Override
        public int estimateCount() {
            return registry.estimateCount();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
        }

        public void completeTask(Status.Code code) {
            taskStatus(code).inc();
        }

        private Rate taskStatus(Status.Code code) {
            var result = status.get(code);
            if (result == null) {
                result = registry.rate("scheduler.tasks.status", Labels.of("status", code.name()));
                this.status.putIfAbsent(code, result);
            }
            return result;
        }

        public void completePipeline(Status.Code code) {
            pipelineStatus(code).inc();
        }

        private Rate pipelineStatus(Status.Code code) {
            var result = pipelineStatus.get(code);
            if (result == null) {
                result = registry.rate("scheduler.tasks.pipeline.status", Labels.of("status", code.name()));
                this.pipelineStatus.putIfAbsent(code, result);
            }

            return result;
        }

        public void spendTime(TaskPipeline.State state, long nanos) {
            pipelineTimeInState.get(state).add(nanos);
        }

        public void combine(Type other) {
            schedule.combine(other.schedule);
            reschedule.combine(other.reschedule);
            rescheduleExternally.combine(other.rescheduleExternally);
            progress.combine(other.progress);
            cancel.combine(other.cancel);
            elapsedMillis.combine(other.elapsedMillis);
            waitToExec.combine(other.waitToExec);
            lagMillis.combine(other.lagMillis);
            pipeline.combine(other.pipeline);

            for (var entry : other.status.entrySet()) {
                taskStatus(entry.getKey()).combine(entry.getValue());
            }

            for (var entry : other.pipelineStatus.entrySet()) {
                pipelineStatus(entry.getKey()).combine(entry.getValue());
            }

            EnumMetrics.combineRates(TaskPipeline.State.class, pipelineTimeInState, other.pipelineTimeInState);
        }
    }
}
