package ru.yandex.solomon.scheduler.context;

import java.time.Clock;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import com.google.common.base.Strings;
import com.google.protobuf.TypeRegistry;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.TaskSchedulerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.ydb.YdbLocksDao;
import ru.yandex.solomon.scheduler.TaskDeps;
import ru.yandex.solomon.scheduler.TaskExecutor;
import ru.yandex.solomon.scheduler.TaskExecutorImpl;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.scheduler.TaskMetrics;
import ru.yandex.solomon.scheduler.TaskScheduler;
import ru.yandex.solomon.scheduler.TaskSchedulerImpl;
import ru.yandex.solomon.scheduler.dao.SchedulerDao;
import ru.yandex.solomon.scheduler.dao.ydb.YdbSchedulerDao;
import ru.yandex.solomon.scheduler.grpc.GrpcSchedulerService;
import ru.yandex.solomon.scheduler.www.TaskScheduledController;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
public class TaskSchedulerContext {
    private final ThreadPoolProvider threads;
    private final String rootPath;
    private final TableClient tableClient;
    private final SchemeClient schemeClient;
    private final TaskSchedulerConfig config;

    public TaskSchedulerContext(String rootPath, TaskSchedulerConfig config, ThreadPoolProvider threads, TableClient tableClient, SchemeClient schemeClient) {
        this.rootPath = rootPath + "/TaskScheduler";
        this.config = config;
        this.threads = threads;
        this.tableClient = tableClient;
        this.schemeClient = schemeClient;
    }

    @Bean
    SchedulerDao schedulerDao(MetricRegistry registry) {
        var dao = new YdbSchedulerDao(rootPath, tableClient, schemeClient);
        dao.createSchema();
        return measure(dao, SchedulerDao.class, registry);
    }

    @Bean
    TaskExecutorImpl executor(List<TaskHandler> handlers) {
        return new TaskExecutorImpl(executor(), handlers);
    }

    @Bean
    TaskDeps deps(TaskExecutor taskExecutor, SchedulerDao schedulerDao, MetricRegistry registry) {
        var timer = threads.getSchedulerExecutorService();
        var ydbLocksDao = new YdbLocksDao(tableClient, Clock.systemUTC(), rootPath);
        var lockDao= measure(ydbLocksDao, LocksDao.class, registry);
        ydbLocksDao.createSchema();
        return new TaskDeps(HostUtils.getFqdn(), taskExecutor, schedulerDao, lockDao, Clock.systemUTC(), executor(), timer);
    }

    @Bean
    public TaskSchedulerImpl taskScheduler(TaskDeps deps) {
        var interval = Optional.of(TimeConverter.protoToDuration(config.getFetchInterval()))
                .filter(duration -> !duration.isZero())
                .orElse(Duration.ofMinutes(1));
        var maxInflight = config.getMaxInflight() > 0
                ? config.getMaxInflight()
                : 100;
        return new TaskSchedulerImpl(interval, maxInflight, deps);
    }

    @Bean
    public TaskMetrics taskMetrics(TaskSchedulerImpl scheduler) {
        return scheduler.metrics();
    }

    @Bean
    public TypeRegistry typeRegistry(List<TaskHandler> handlers) {
        var result = TypeRegistry.newBuilder();
        for (var handler : handlers) {
            for (var descriptor : handler.descriptors()) {
                result.add(descriptor);
            }
        }

        return result.build();
    }

    @Bean
    public RootLink taskSchedulerLinks() {
        return new RootLink("/task-scheduler", "Task Scheduler");
    }

    @Bean
    public TaskScheduledController taskScheduledController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            TypeRegistry registry,
            SchedulerDao dao)
    {
        return new TaskScheduledController(authenticator, authorizer, registry, dao);
    }

    @Bean
    public GrpcSchedulerService grpcSchedulerService(TaskScheduler scheduler) {
        return new GrpcSchedulerService(scheduler);
    }

    protected Executor executor() {
        var threadPoolName = !Strings.isNullOrEmpty(config.getThreadPoolName())
                ? config.getThreadPoolName()
                : "CpuLowPriority";
        return threads.getExecutorService(threadPoolName, "ThreadPoolName");
    }

    protected ScheduledExecutorService timer() {
        return threads.getSchedulerExecutorService();
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass, MetricRegistry registry) {
        return DaoMetricsProxy.of(dao, daoClass, registry);
    }
}
