package ru.yandex.solomon.scheduler.dao.ydb;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.function.Consumer;

import com.google.common.base.Strings;
import com.google.protobuf.Any;
import com.google.protobuf.ByteString;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import io.grpc.Status;
import io.grpc.Status.Code;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.scheduler.ScheduledTask;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.Task.State;
import ru.yandex.solomon.scheduler.dao.SchedulerDao;
import ru.yandex.solomon.scheduler.dao.ydb.YdbSchedulerTable.ScheduledTaskReader;
import ru.yandex.solomon.scheduler.dao.ydb.YdbSchedulerTable.TaskColumnReader;
import ru.yandex.solomon.scheduler.dao.ydb.YdbSchedulerTable.TaskReader;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.string;
import static com.yandex.ydb.table.values.PrimitiveValue.timestamp;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.uint64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint8;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbSchedulerDao implements SchedulerDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbSchedulerDao.class);

    private final SessionRetryContext retryCtx;
    private final SchemeClient schemeClient;
    private final YdbSchedulerQuery query;

    public YdbSchedulerDao(String root, TableClient tableClient, SchemeClient schemeClient) {
        this.schemeClient = schemeClient;
        this.query = new YdbSchedulerQuery(root);
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();
    }

    @Override
    public CompletableFuture<Void> createSchema() {
        return schemeClient.makeDirectories(query.root)
                .thenAccept(status -> status.expect("parent directories success created"))
                .thenCompose(unused -> createTable(query.tableTasks, YdbSchedulerTable::createTasksTable))
                .thenCompose(unused -> createTable(query.tableScheduled, YdbSchedulerTable::createScheduledTasksTable));
    }

    private CompletableFuture<Void> createTable(String tablePath, BiFunction<String, Session, CompletableFuture<com.yandex.ydb.core.Status>> fn) {
        return schemeClient.describePath(tablePath)
                .thenCompose(exist -> {
                    if (exist.isSuccess()) {
                        return completedFuture(com.yandex.ydb.core.Status.SUCCESS);
                    }

                    return retryCtx.supplyStatus(session -> fn.apply(tablePath, session));
                })
                .thenAccept(status -> status.expect("cannot create create table " + tablePath));
    }

    @Override
    public CompletableFuture<Boolean> add(Task task) {
        var params = Params.create()
                .put("$hash", uint32(task.id().hashCode()))
                .put("$id", utf8(task.id()))
                .put("$type", utf8(task.type()))
                .put("$params", string(task.params().toByteString()))
                .put("$execute_at", timestamp(Instant.ofEpochMilli(task.executeAt())))
                .put("$version", uint32(task.version()));

        return execute(query.insert, params)
                .thenApply(result -> {
                    if (result.isSuccess()) {
                        return Boolean.TRUE;
                    }

                    if (result.getCode() == StatusCode.PRECONDITION_FAILED) {
                        return Boolean.FALSE;
                    }

                    result.expect("unable insert new task");
                    return Boolean.FALSE;
                });
    }

    @Override
    public CompletableFuture<Optional<Task>> get(String taskId) {
        Params params = Params.of("$hash", uint32(taskId.hashCode()), "$id", utf8(taskId));
        return execute(query.selectOne, params)
                .thenApply(result -> {
                    var dataQueryResult = result.expect("unable select task by id");
                    var rs = dataQueryResult.getResultSet(0);
                    if (!rs.next()) {
                        return Optional.empty();
                    }

                    var reader = new TaskColumnReader(rs);
                    return Optional.of(reader.read(rs));
                });
    }

    @Override
    public CompletableFuture<Boolean> changeState(String taskId, State state, long seqNo) {
        var params = Params.create()
                .put("$hash", uint32(taskId.hashCode()))
                .put("$id", utf8(taskId))
                .put("$seq_no", uint64(seqNo))
                .put("$state", uint8((byte) state.number));

        return execute(query.changeState, params)
                .thenApply(this::isSuccessUpdateQuery);
    }

    @Override
    public CompletableFuture<Boolean> complete(String taskId, Any result, long seqNo) {
        var params = Params.create()
                .put("$hash", uint32(taskId.hashCode()))
                .put("$id", utf8(taskId))
                .put("$seq_no", uint64(seqNo))
                .put("$result", string(result.toByteString()))
                .put("$status_code", int32(Code.OK.value()))
                .put("$status_description", utf8(""))
                .put("$now", timestamp(Instant.now()));

        return execute(query.complete, params)
                .thenApply(this::isSuccessUpdateQuery);
    }

    @Override
    public CompletableFuture<Boolean> failed(String taskId, Status status, long seqNo) {
        var params = Params.create()
                .put("$hash", uint32(taskId.hashCode()))
                .put("$id", utf8(taskId))
                .put("$seq_no", uint64(seqNo))
                .put("$result", string(ByteString.EMPTY))
                .put("$status_code", int32(status.getCode().value()))
                .put("$status_description", utf8(Strings.nullToEmpty(status.getDescription())))
                .put("$now", timestamp(Instant.now()));

        return execute(query.complete, params)
                .thenApply(this::isSuccessUpdateQuery);
    }

    @Override
    public CompletableFuture<Boolean> reschedule(String taskId, long executeAt, Any progress, long seqNo) {
        var params = Params.create()
                .put("$hash", uint32(taskId.hashCode()))
                .put("$id", utf8(taskId))
                .put("$seq_no", uint64(seqNo))
                .put("$progress", string(progress.toByteString()))
                .put("$execute_at", timestamp(Instant.ofEpochMilli(executeAt)));

        return execute(query.reschedule, params)
                .thenApply(this::isSuccessUpdateQuery);
    }

    @Override
    public CompletableFuture<Boolean> rescheduleExternally(
        String taskId,
        long executeAt,
        Any progress,
        int expectedVersion)
    {
        var params = Params.create()
            .put("$hash", uint32(taskId.hashCode()))
            .put("$id", utf8(taskId))
            .put("$progress", string(progress.toByteString()))
            .put("$execute_at", timestamp(Instant.ofEpochMilli(executeAt)))
            .put("$version", uint32(expectedVersion));

        return execute(query.rescheduleExternally, params)
            .thenApply(this::isSuccessUpdateQuery);
    }

    @Override
    public CompletableFuture<Boolean> progress(String taskId, Any progress, long seqNo) {
        var params = Params.create()
                .put("$hash", uint32(taskId.hashCode()))
                .put("$id", utf8(taskId))
                .put("$seq_no", uint64(seqNo))
                .put("$progress", string(progress.toByteString()));

        return execute(query.progress, params)
                .thenApply(this::isSuccessUpdateQuery);
    }

    @Override
    public CompletableFuture<Void> list(Consumer<Task> consumer) {
        var reader = new TaskReader(consumer);
        return retryCtx.supplyStatus(session -> {
            var settings = ReadTableSettings.newBuilder()
                    .timeout(1, TimeUnit.MINUTES)
                    .orderedRead(true);

            var lastKey = reader.lastKey();
            if (lastKey != null) {
                settings.fromKeyExclusive(lastKey);
            }

            return session.readTable(query.tableTasks, settings.build(), reader);
        }).thenAccept(status -> status.expect("can't read " + query.tableTasks));
    }

    @Override
    public CompletableFuture<List<ScheduledTask>> listScheduled(long executeAt, int limit) {
        return listScheduledBySelect(executeAt, limit)
                .handle((result, e) -> {
                    if (e != null) {
                        logger.warn("Unable find resource by select, fallback to readtable", e);
                        return listScheduledByReadTable(executeAt, limit);
                    }

                    if (result == null) {
                        // more records that limit for select
                        return listScheduledByReadTable(executeAt, limit);
                    }

                    return completedFuture(result.subList(0, Math.min(result.size(), limit)));
                })
                .thenCompose(future -> future);
    }

    private CompletableFuture<List<ScheduledTask>> listScheduledBySelect(long executeAt, int limit) {
        return execute(query.scheduled, Params.of("$execute_at", timestamp(Instant.ofEpochMilli(executeAt))))
                .thenApply(result -> {
                    DataQueryResult dataQueryResult = result.expect("cannot select scheduled tasks");
                    ResultSetReader resultSet = dataQueryResult.getResultSet(0);
                    if (limit <= YdbSchedulerQuery.SELECT_LIMIT) {
                        var reader = new ScheduledTaskReader();
                        reader.accept(resultSet);
                        return reader.list.subList(0, limit);
                    }

                    if (resultSet.isTruncated() || resultSet.getRowCount() >= YdbSchedulerQuery.SELECT_LIMIT) {
                        return null;
                    }

                    var reader = new ScheduledTaskReader();
                    reader.accept(resultSet);
                    return reader.list;
                });
    }

    private CompletableFuture<List<ScheduledTask>> listScheduledByReadTable(long executeAt, int limit) {
        var reader = new ScheduledTaskReader();
        var toKey = timestamp(Instant.ofEpochMilli(executeAt));
        return retryCtx.supplyStatus(session -> {
            var settings = ReadTableSettings.newBuilder()
                    .timeout(1, TimeUnit.MINUTES)
                    .orderedRead(true)
                    .rowLimit(limit)
                    .toKeyInclusive(toKey);

            var lastKey = reader.lastKey();
            if (lastKey != null) {
                settings.fromKeyExclusive(lastKey);
            }

            return session.readTable(query.tableScheduled, settings.build(), reader);
        }).thenApply(status -> {
            status.expect("can't read " + query.tableScheduled);
            return reader.list;
        });
    }

    private boolean isSuccessUpdateQuery(Result<DataQueryResult> result) {
        var dataQueryResult = result.expect("unable change task state");
        var rs = dataQueryResult.getResultSet(0);
        if (rs.next()) {
            return rs.getColumn(0).getBool();
        }

        return false;
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return retryCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }
}
