package ru.yandex.solomon.scheduler.dao.ydb;

import java.time.Instant;
import java.util.ArrayList;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AutoPartitioningPolicy;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.PartitioningPolicy;
import com.yandex.ydb.table.settings.TtlSettings;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleValue;

import ru.yandex.solomon.scheduler.ScheduledTask;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.Task.State;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbSchedulerTable {

    static CompletableFuture<Status> createTasksTable(String tablePath, Session session) {
        var table = TableDescription.newBuilder()
                .addNullableColumn("hash", PrimitiveType.uint32())
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("type", PrimitiveType.utf8())
                .addNullableColumn("params", PrimitiveType.string())
                .addNullableColumn("execute_at", PrimitiveType.timestamp())
                .addNullableColumn("state", PrimitiveType.uint8())
                .addNullableColumn("status_code", PrimitiveType.int32())
                .addNullableColumn("status_description", PrimitiveType.utf8())
                .addNullableColumn("progress", PrimitiveType.string())
                .addNullableColumn("result", PrimitiveType.string())
                .addNullableColumn("completed_at", PrimitiveType.timestamp())
                .addNullableColumn("seq_no", PrimitiveType.uint64())
                .addNullableColumn("version", PrimitiveType.uint32())
                .setPrimaryKeys("hash", "id")
                .build();

        var settings = new CreateTableSettings();
        settings.setTtlSettings(new TtlSettings("completed_at", (int) TimeUnit.DAYS.toSeconds(7)));
        settings.setPartitioningPolicy(new PartitioningPolicy().setAutoPartitioning(AutoPartitioningPolicy.AUTO_SPLIT_MERGE));
        settings.setTimeout(10, TimeUnit.SECONDS);

        return session.createTable(tablePath, table, settings);
    }

    static CompletableFuture<Status> createScheduledTasksTable(String tablePath, Session session) {
        var table = TableDescription.newBuilder()
                .addNullableColumn("execute_at", PrimitiveType.timestamp())
                .addNullableColumn("id", PrimitiveType.utf8())
                .addNullableColumn("type", PrimitiveType.utf8())
                .addNullableColumn("params", PrimitiveType.string())
                .setPrimaryKeys("execute_at", "id")
                .build();

        var settings = new CreateTableSettings();
        settings.setPartitioningPolicy(new PartitioningPolicy().setAutoPartitioning(AutoPartitioningPolicy.DISABLED));
        settings.setTimeout(10, TimeUnit.SECONDS);

        return session.createTable(tablePath, table, settings);
    }

    private static Any any(ResultSetReader rs, int idx) {
        try {
            return Any.parseFrom(rs.getColumn(idx).getString());
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to parse column " + rs.getColumnName(idx) + " at row " + rs.getRowCount());
        }
    }

    private static String utf8(ResultSetReader rs, int idx) {
        return rs.getColumn(idx).getUtf8();
    }

    private static long timestamp(ResultSetReader rs, int idx) {
        return rs.getColumn(idx).getTimestamp().toEpochMilli();
    }

    private static State state(ResultSetReader rs, int idx) {
        return State.valueOf(rs.getColumn(idx).getUint8());
    }

    private static int int32(ResultSetReader rs, int idx) {
        return rs.getColumn(idx).getInt32();
    }

    private static int uint32(ResultSetReader rs, int idx) {
        return (int) rs.getColumn(idx).getUint32();
    }

    static class TaskColumnReader {
        private final int idIdx;
        private final int typeIdx;
        private final int paramsIdx;
        private final int executeAtIdx;
        private final int stateIdx;
        private final int statusCodeIdx;
        private final int statusDescriptionIdx;
        private final int progressIdx;
        private final int resultIdx;
        private final int versionIdx;

        public TaskColumnReader(ResultSetReader resultSet) {
            this.idIdx = resultSet.getColumnIndex("id");
            this.typeIdx = resultSet.getColumnIndex("type");
            this.paramsIdx = resultSet.getColumnIndex("params");
            this.executeAtIdx = resultSet.getColumnIndex("execute_at");
            this.stateIdx = resultSet.getColumnIndex("state");
            this.statusCodeIdx = resultSet.getColumnIndex("status_code");
            this.statusDescriptionIdx = resultSet.getColumnIndex("status_description");
            this.progressIdx = resultSet.getColumnIndex("progress");
            this.resultIdx = resultSet.getColumnIndex("result");
            this.versionIdx = resultSet.getColumnIndex("version");
        }

        public Task read(ResultSetReader rs) {
            var task = Task.newBuilder();
            task.setId(utf8(rs, idIdx));
            task.setType(utf8(rs, typeIdx));
            task.setParams(any(rs, paramsIdx));
            task.setExecuteAt(timestamp(rs, executeAtIdx));
            task.setState(state(rs, stateIdx));
            task.setStatus(io.grpc.Status.fromCodeValue(int32(rs, statusCodeIdx))
                    .withDescription(Strings.emptyToNull(utf8(rs, statusDescriptionIdx))));
            task.setProgress(any(rs, progressIdx));
            task.setResult(any(rs, resultIdx));
            task.setVersion(uint32(rs, versionIdx));
            return task.build();
        }
    }

    static class TaskReader implements Consumer<ResultSetReader> {
        private final Consumer<Task> consumer;
        @Nullable
        private Task last;

        public TaskReader(Consumer<Task> consumer) {
            this.consumer = consumer;
        }

        @Override
        public void accept(ResultSetReader rs) {
            int rowCount = rs.getRowCount();
            if (rowCount == 0) {
                return;
            }

            var columns = new TaskColumnReader(rs);
            while (rs.next()) {
                last = columns.read(rs);
                consumer.accept(last);
            }
        }

        @Nullable
        public TupleValue lastKey() {
            if (last == null) {
                return null;
            }

            return TupleValue.of(
                    PrimitiveValue.uint32(last.id().hashCode()).makeOptional(),
                    PrimitiveValue.utf8(last.id()).makeOptional());
        }
    }

    static class ScheduledTaskColumnReader {
        private final int idIdx;
        private final int typeIdx;
        private final int executeAtIdx;
        private final int paramsIdx;

        public ScheduledTaskColumnReader(ResultSetReader resultSet) {
            this.idIdx = resultSet.getColumnIndex("id");
            this.typeIdx = resultSet.getColumnIndex("type");
            this.executeAtIdx = resultSet.getColumnIndex("execute_at");
            this.paramsIdx = resultSet.getColumnIndex("params");
        }

        public ScheduledTask read(ResultSetReader rs) {
            return new ScheduledTask(timestamp(rs, executeAtIdx), utf8(rs, idIdx), utf8(rs, typeIdx), any(rs, paramsIdx));
        }
    }

    static class ScheduledTaskReader implements Consumer<ResultSetReader> {
        final ArrayList<ScheduledTask> list = new ArrayList<>();

        @Override
        public void accept(ResultSetReader rs) {
            int rowCount = rs.getRowCount();
            if (rowCount == 0) {
                return;
            }

            list.ensureCapacity(list.size() + rowCount);

            var columns = new ScheduledTaskColumnReader(rs);
            while (rs.next()) {
                list.add(columns.read(rs));
            }
        }

        @Nullable
        public TupleValue lastKey() {
            if (list.isEmpty()) {
                return null;
            }

            var last = list.get(list.size() - 1);

            return TupleValue.of(
                    PrimitiveValue.timestamp(Instant.ofEpochMilli(last.executeAt())).makeOptional(),
                    PrimitiveValue.utf8(last.id()).makeOptional());
        }
    }
}
