package ru.yandex.solomon.scheduler.www;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.protobuf.TypeRegistry;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.Account;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.scheduler.dao.SchedulerDao;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
public class TaskScheduledController {
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final TypeRegistry registry;
    private final SchedulerDao dao;

    @Autowired
    public TaskScheduledController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            TypeRegistry registry,
            SchedulerDao dao)
    {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.registry = registry;
        this.dao = dao;
    }

    @RequestMapping(value = "/task-scheduler", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> tasks(ServerHttpRequest request, @RequestParam Map<String, String> params) {
        return authorize(request)
                .thenCompose(ignore -> {
                    var page = new TaskSchedulerPage("/task-scheduler", params, registry);
                    return dao.listScheduled(page.filterUntilMillis(), 10_000)
                            .thenApply(scheduled -> {
                                page.onLoad(scheduled);
                                return page.genString();
                            });
                });
    }

    @RequestMapping(value = "/task-scheduler/{id}", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> taskById(ServerHttpRequest request, @PathVariable("id") String id) {
        return authorize(request)
                .thenCompose(ignore -> dao.get(id))
                .thenApply(task -> {
                    if (task.isEmpty()) {
                        return "task with id " + id + " not found";
                    }

                    return new TaskPage(task.get(), registry).genString();
                });
    }

    private CompletableFuture<Account> authorize(ServerHttpRequest request) {
        return authenticator.authenticate(request)
                .thenCompose(authorizer::authorize);
    }
}
