package ru.yandex.solomon.scheduler.www;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import com.google.protobuf.TextFormat;
import com.google.protobuf.TypeRegistry;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.scheduler.ScheduledTask;
import ru.yandex.solomon.staffOnly.UrlUtils;
import ru.yandex.solomon.staffOnly.html.CssLine;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.WritableToHtml;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class TaskSchedulerPage extends ManagerPageTemplate {
    private static final String FILTER_BY_ID_PARAM = "filterById";
    private static final String FILTER_BY_TYPE_PARAM = "filterByType";
    private static final String FILTER_BY_MIN_DELAY_PARAM = "filterByMinDelay";
    private static final String FILTER_BY_MAX_DELAY_PARAM = "filterByMaxDelay";
    private static final String FILTER_BY_PARAMS_PARAM = "filterByParams";
    private static final String SORT_BY_COLUMN_PARAM = "sortBy";
    private static final String LIMIT_PARAM = "limit";

    private final String path;
    private final Map<String, String> params;
    private final TypeRegistry registry;
    private final long now;

    // Filters
    private String filterById;
    private String filterByType;
    private long filterByMinDelay;
    private long filterByMaxDelay;
    private String filterByParams;

    // Sorts
    private String sortByColumn;
    private boolean sortAsAsc;

    // Limits
    private int limit;

    private final List<Record> records = new ArrayList<>();
    private List<Column<Record>> columns;

    public TaskSchedulerPage(String path, Map<String, String> params, TypeRegistry registry) {
        super("Task Scheduler");
        this.path = path;
        this.params = params;
        this.registry = registry;
        this.now = System.currentTimeMillis();
        parseParameters();
        prepareColumn();
    }

    public void onLoad(List<ScheduledTask> tasks) {
        var filter = prepareFilterPredicate();
        for (var task : tasks) {
            var record = Record.of(task, registry, now);
            if (filter.test(record)) {
                records.add(record);
            }
        }

        var sorting = prepareSorting();
        if (sorting != null) {
            records.sort(sorting);
        }
    }

    public long filterUntilMillis() {
        return now + filterByMaxDelay;
    }

    @Override
    protected void content() {
        tag("div.row", () -> {
            tag("div.col-md-6", () -> {
                tag("div.well", this::writeFiltersForm);
            });
        });
        writeTable();
    }

    private void parseParameters() {
        this.filterById = getParam(FILTER_BY_ID_PARAM);
        this.filterByType = getParam(FILTER_BY_TYPE_PARAM);
        this.filterByMinDelay = DurationUtils
                .parseDuration(getParam(FILTER_BY_MIN_DELAY_PARAM))
                .orElse(Duration.ZERO)
                .toMillis();
        this.filterByMaxDelay = DurationUtils
                .parseDuration(getParam(FILTER_BY_MAX_DELAY_PARAM))
                .orElse(Duration.ofHours(1))
                .toMillis();
        this.filterByParams = getParam(FILTER_BY_PARAMS_PARAM);
        {
            var str = getParam(LIMIT_PARAM);
            this.limit = Strings.isNullOrEmpty(str) ? 100 : Integer.parseInt(str);
        }
        {
            String sortByStr = getParam(SORT_BY_COLUMN_PARAM);
            this.sortAsAsc = !sortByStr.startsWith("-");
            this.sortByColumn = sortByStr.startsWith("-") ? sortByStr.substring(1) : sortByStr;
        }
    }

    private String getParam(String name) {
        return params.getOrDefault(name, "").trim();
    }

    private Predicate<Record> prepareFilterPredicate() {
        Predicate<Record> filter = ignore -> true;
        if (!filterById.isEmpty()) {
            var selector = selector("id", filterById);
            filter = filter.and(record -> selector.match(record.id));
        }

        if (!filterByType.isEmpty()) {
            var selector = selector("type", filterByType);
            filter = filter.and(record -> selector.match(record.type));
        }

        if (filterByMinDelay != 0) {
            filter = filter.and(record -> record.delay >= filterByMinDelay);
        }

        if (!filterByParams.isEmpty()) {
            var selector = selector("params", filterByParams);
            filter = filter.and(record -> selector.match(record.params));
        }

        return filter;
    }

    private Selector selector(String key, String value) {
        if (value.startsWith("!")) {
            return Selector.notGlob(key, value.substring(1));
        }
        return Selector.glob(key, value);
    }

    @Nullable
    private Comparator<Record> prepareSorting() {
        if (sortByColumn.isEmpty()) {
            return null;
        }

        var sortBy = columns.stream()
                .filter(column -> column.title.equals(sortByColumn))
                .findFirst()
                .orElse(null);

        if (sortBy == null) {
            return null;
        }

        if (sortAsAsc) {
            return sortBy.comparator;
        } else {
            return sortBy.comparator.reversed();
        }
    }

    private void prepareColumn() {
        List<Column<Record>> result = new ArrayList<>();
        result.add(Column.of(
                "ExecuteAt",
                r -> Instant.ofEpochMilli(r.executeAt()).toString(),
                Comparator.comparingLong(Record::executeAt)));
        result.add(Column.of(
                "Delay",
                r -> DurationUtils.formatDurationMillisTruncated(r.delay()),
                Comparator.comparingLong(Record::delay)));
        result.add(Column.of(
                "Type",
                Record::type,
                Comparator.comparing(Record::type)));
        result.add(Column.of(
                "Id",
                r -> (WritableToHtml) mw -> {
                    var hw = mw.getHtmlWriter();
                    hw.aHref(path + "/" + r.id(), r.id());
        },
                Comparator.comparing(Record::id)));
        result.add(Column.of(
                "Params",
                r -> (WritableToHtml) mw -> {
                    mw.getHtmlWriter().preText(r.params);
                },
                Comparator.comparing(Record::params)));
        this.columns = result;
    }

    private void writeFiltersForm() {
        form(() -> {
            tag("div.form-group", () -> {
                h3("Filters");
                table(() -> {
                    textInput(
                            FILTER_BY_ID_PARAM,
                            "Id:",
                            filterById,
                            "glob");
                    textInput(
                            FILTER_BY_TYPE_PARAM,
                            "Type:",
                            filterByType,
                            "glob");
                    textInput(
                            FILTER_BY_MIN_DELAY_PARAM,
                            "Min delay:",
                            getParam(FILTER_BY_MIN_DELAY_PARAM),
                            "duration");
                    textInput(
                            FILTER_BY_MAX_DELAY_PARAM,
                            "Max delay:",
                            DurationUtils.formatDurationMillis(filterByMaxDelay),
                            "duration");
                    textInput(
                            FILTER_BY_PARAMS_PARAM,
                            "Params:",
                            filterByParams,
                            "glob");
                    textInput(
                            LIMIT_PARAM,
                            "Limit:",
                            Integer.toString(limit),
                            "number");
                });
            });
            tag("div.form-group", () -> {
                buttonSubmitDefault("Apply");
                write(" ");
                tag("a.btn.btn-warning", new Attr("href", path), () -> write("Clear Filters"));
            });
        }, new Attr("action", path), new Attr("class", "form-horizontal"));
    }

    private void textInput(String id, String name, String value, String help) {
        tr(() -> {
            thText(name);
            td(() -> {
                inputTextfieldFormControl(id, id, value);
            });
            if (StringUtils.isNotEmpty(help)) {
                td(() -> {
                    tag("span", HtmlWriter.Attr.cssClass("help-block"), () -> {
                        write(help);
                    });
                });
            } else {
                td(() -> {
                });
            }
        });
    }

    private void writeTable() {
        var writer = new ManagerWriter(new ManagerWriterContext(), this);
        div(() -> write("Records: " + records.size()));
        if (records.size() > limit) {
            div(() -> write("Table truncated, add one more filter"));
        }
        tag("table", () -> {
            tr(() -> {
                for (var column : columns) {
                    th(() -> {
                        String title = column.title;
                        if (!title.equals(sortByColumn)) {
                            aHref(hrefWithParam(SORT_BY_COLUMN_PARAM, title), title);
                        } else {
                            String link = sortAsAsc ? "-" + title : title;
                            String icon = sortAsAsc
                                    ? "glyphicon-sort-by-attributes"
                                    : "glyphicon-sort-by-attributes-alt";
                            aHref(hrefWithParam(SORT_BY_COLUMN_PARAM, link), () -> {
                                write(column.title);
                                tag("span", HtmlWriter.Attr.cssClass("glyphicon " + icon));
                            });
                        }
                    });
                }
            });
            int recordCount = 0;
            for (var record : records) {
                if (recordCount++ >= limit) {
                    break;
                }
                tr(() -> {
                    for (var column : columns) {
                        writer.writeCellTd(column.content.apply(record));
                    }
                });
            }
        }, Attr.cssClass("table simple-table2 table-hover table-condensed"), Attr.style(new CssLine("word-break", "break-all")));
    }

    private String hrefWithParam(String name, String value) {
        return UrlUtils.hrefWithParam(name, value, path, params);
    }

    private static record Record(long executeAt, long delay, String id, String type, String params) implements TableRecord {
        public static Record of(ScheduledTask task, TypeRegistry registry, long now) {
            var params = TextFormat.printer()
                    .usingTypeRegistry(registry)
                    .printToString(task.params());
            return new Record(task.executeAt(), task.executeAt() - now, task.id(), task.type(), params);
        }
    }

}
