package ru.yandex.solomon.scheduler;

import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.solomon.scheduler.ExecutionContextStub.Cancel;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Complete;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Event;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Fail;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Progress;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Reschedule;
import ru.yandex.solomon.scheduler.handlers.Tasks;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class ExecutionContextStubTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(30, TimeUnit.SECONDS)
            .build();

    @Test
    public void autoCompleteComplete() {
        var context = context();
        var any = Tasks.anyAsNumber();

        context.complete(any).join();

        var event = context.takeEvent(Complete.class);
        assertEquals(any, event.result());
        assertTrue(event.future().isDone());

        assertDoneFuture(event, context);
    }

    @Test
    public void autoCompleteFail() {
        var context = context();

        var exception = new RuntimeException("hi");
        context.fail(exception).join();

        var event = context.takeEvent(Fail.class);
        assertEquals(exception, event.throwable());
        assertTrue(event.future().isDone());

        assertDoneFuture(event, context);
    }

    @Test
    public void autoCompleteReschedule() {
        var context = context();

        var time = ThreadLocalRandom.current().nextLong();
        var progress = Tasks.anyAsNumber();
        context.reschedule(time, progress).join();

        var event = context.takeEvent(Reschedule.class);
        assertEquals(time, event.executeAt());
        assertEquals(progress, event.progress());
        assertTrue(event.future().isDone());

        assertDoneFuture(event, context);
    }

    @Test
    public void autoCompleteCancel() {
        var context = context();

        context.cancel().join();

        var event = context.takeEvent(Cancel.class);
        assertTrue(event.future().isDone());

        assertDoneFuture(event, context);
    }

    @Test
    public void autoCompleteProgress() {
        var context = context();
        var any = Tasks.anyAsNumber();

        context.progress(any).join();

        var event = context.takeEvent(Progress.class);
        assertEquals(any, event.progress());
        assertTrue(event.future().isDone());

        var doneFuture = context.expectDone(Event.class);
        assertFalse(doneFuture.isDone());
    }

    @Test
    public void complete() {
        var context = context();
        context.autoComplete = false;
        var any = Tasks.anyAsNumber();

        var waitFuture = context.complete(any);

        var event = context.takeEvent(Complete.class);
        assertFalse(event.future().isDone());
        assertFalse(waitFuture.isDone());
        assertFalse(context.expectDone(Event.class).isCompletedExceptionally());
        assertEquals(any, event.result());

        event.future().completeExceptionally(new Throwable("hi"));
        assertTrue(event.future().isCompletedExceptionally());
        assertTrue(waitFuture.isCompletedExceptionally());
        assertTrue(context.expectDone(Event.class).isCompletedExceptionally());
    }

    private void assertDoneFuture(Event expect, ExecutionContextStub context) {
        var doneFuture = context.expectDone(Event.class);
        assertTrue(doneFuture.isDone());
        assertEquals(expect, doneFuture.join());
    }

    public ExecutionContextStub context() {
        return new ExecutionContextStub(Tasks.randomTask());
    }
}
