package ru.yandex.solomon.scheduler;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.protobuf.Any;
import com.google.protobuf.Int32Value;
import com.google.protobuf.StringValue;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Complete;
import ru.yandex.solomon.scheduler.handlers.Tasks;
import ru.yandex.solomon.util.Proto;

import static java.util.stream.Collectors.collectingAndThen;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.scheduler.handlers.Tasks.concatTask;
import static ru.yandex.solomon.scheduler.handlers.Tasks.decTask;
import static ru.yandex.solomon.scheduler.handlers.Tasks.incTask;

/**
 * @author Vladimir Gordiychuk
 */
public class TaskExecutorImplTest {

    private TaskExecutor executor;

    @Before
    public void setUp() {
        executor = new TaskExecutorImpl(ForkJoinPool.commonPool(), Tasks.handlers());
    }

    @Test
    public void notPermitUnknownType() {
        assertNull(executor.acquire("id", "unknown_type", Any.getDefaultInstance()));
    }

    @Test
    public void permitKnownType() {
        assertNotNull(executor.acquire("id", "number", Any.getDefaultInstance()));
        assertNotNull(executor.acquire("id", "string", Any.getDefaultInstance()));
    }

    @Test
    public void execDifferentArg() {
        assertEquals(1, Proto.unpack(exec(incTask()).join(), Int32Value.class).getValue());
        assertEquals(2, Proto.unpack(exec(incTask()).join(), Int32Value.class).getValue());
        assertEquals(3, Proto.unpack(exec(incTask()).join(), Int32Value.class).getValue());
        assertEquals(2, Proto.unpack(exec(decTask()).join(), Int32Value.class).getValue());
    }

    @Test
    public void execDifferentType() {
        assertEquals("test", Proto.unpack(exec(concatTask("test")).join(), StringValue.class).getValue());
    }

    @Test
    public void parallelExec() {
        var tasks = IntStream.range(0, 100)
                .mapToObj(ignore -> incTask())
                .collect(Collectors.toList());

        int lastState = tasks.parallelStream()
                .map(this::exec)
                .collect(collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
                .join()
                .stream()
                .mapToInt(any -> Proto.unpack(any, Int32Value.class).getValue())
                .max()
                .orElse(0);

        assertEquals(100, lastState);
    }

    private CompletableFuture<Any> exec(Task task) {
        var context = new ExecutionContextStub(task);
        executor.execute(context);
        return context.expectDone(Complete.class)
                .thenApply(Complete::result);
    }
}
