package ru.yandex.solomon.scheduler;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.protobuf.Any;
import com.google.protobuf.Int32Value;
import com.google.protobuf.Message;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.scheduler.ExecutionContextStub.Complete;
import ru.yandex.solomon.util.Proto;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
public class TaskExecutorStub implements TaskExecutor {
    private final Map<String, TaskHandler> handlerByType;
    private final ConcurrentMap<String, CompletableFuture<Any>> taskToFuture = new ConcurrentHashMap<>();

    public TaskExecutorStub(List<TaskHandler> handlers) {
        this.handlerByType = handlers.stream().collect(Collectors.toMap(TaskHandler::type, Function.identity()));
    }

    @Nullable
    @Override
    public Permit acquire(String id, String type, Any params) {
        var handler = handlerByType.get(type);
        if (handler == null) {
            return null;
        }

        return handler.acquire(id, params);
    }

    @Override
    public void execute(ExecutionContext context) {
        var handler = requireNonNull(handlerByType.get(context.task().type()));
        var contextWrapper = new Context(context);
        CompletableFuture.runAsync(() -> {
            try {
                handler.execute(contextWrapper);
            } catch (Throwable e) {
                contextWrapper.fail(e);
            }
        });

        var doneFuture = whenComplete(context.task().id());
        CompletableFutures.whenComplete(contextWrapper.future, doneFuture);
    }

    public CompletableFuture<Any> run(Task task) {
        var context = new ExecutionContextStub(task);
        execute(context);
        return context.expectDone(Complete.class)
                .thenApply(Complete::result);
    }

    public CompletableFuture<Integer> runNumber(Task task) {
        return run(task).thenApply(any -> Proto.unpack(any, Int32Value.class).getValue());
    }

    public CompletableFuture<Any> whenComplete(String taskId) {
        return taskToFuture.computeIfAbsent(taskId, ignore -> new CompletableFuture<>());
    }

    private static class Context implements ExecutionContext {
        private final CompletableFuture<Any> future = new CompletableFuture<>();
        private final ExecutionContext context;

        public Context(ExecutionContext context) {
            this.context = context;
        }

        @Override
        public Task task() {
            return context.task();
        }

        @Override
        public boolean isDone() {
            return context.isDone();
        }

        @Override
        public <T extends Message> CompletableFuture<?> complete(T result) {
            var doneFuture = context.complete(result).thenApply(ignore -> Proto.pack(result));
            CompletableFutures.whenComplete(doneFuture, future);
            return doneFuture;
        }

        @Override
        public CompletableFuture<?> fail(Throwable e) {
            return context.fail(e).whenComplete((ignore, ignore2) -> {
                future.completeExceptionally(e);
            });
        }

        @Override
        public <T extends Message> CompletableFuture<?> reschedule(long executeAt, T progress) {
            return context.reschedule(executeAt, progress).whenComplete((ignore, ignore2) -> {
                future.complete(Any.getDefaultInstance());
            });
        }

        @Override
        public <T extends Message> CompletableFuture<?> progress(T progress) {
            return context.progress(progress);
        }

        @Override
        public CompletableFuture<?> cancel() {
            return context.cancel();
        }
    }
}
