package ru.yandex.solomon.scheduler.dao;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Strings;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.protobuf.Any;
import io.grpc.Status;
import org.junit.Test;

import ru.yandex.solomon.scheduler.ScheduledTask;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.Task.State;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.scheduler.handlers.Tasks.anyAsNumber;
import static ru.yandex.solomon.scheduler.handlers.Tasks.randomTask;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class SchedulerDaoTest {
    protected abstract SchedulerDao getDao();

    @Test
    public void getNotExist() {
        var dao = getDao();
        var result = dao.get("not_exist_task_id").join();
        assertEquals(Optional.empty(), result);
    }

    @Test
    public void getById() {
        var tasks = IntStream.range(0, 3)
                .mapToObj(ignore -> randomTask())
                .collect(Collectors.toList());

        var dao = getDao();
        for (var task : tasks) {
            assertTrue(dao.add(task).join());
        }

        for (var task : tasks) {
            assertTaskEqual(task);
        }
    }

    @Test
    public void addOnlyOnce() {
        var dao = getDao();
        var expected = randomTask();
        assertTrue(dao.add(expected).join());
        assertTaskEqual(expected);

        var duplicate = expected.toBuilder().setExecuteAt(expected.executeAt() + TimeUnit.DAYS.toMillis(1)).build();
        assertFalse(dao.add(duplicate).join());
        assertTaskEqual(expected);
    }

    @Test
    public void changeState() {
        var dao = getDao();
        var init = randomTask();
        assertTrue(dao.add(init).join());

        {
            var expected = init.toBuilder().setState(State.RUNNING).build();
            assertTrue(dao.changeState(init.id(), State.RUNNING, 42L).join());
            assertTaskEqual(expected);
        }
        {
            var expected = init.toBuilder().setState(State.SCHEDULED).build();
            assertTrue(dao.changeState(init.id(), State.SCHEDULED, 42L).join());
            assertTaskEqual(expected);
        }
    }

    @Test
    public void changeStateNotExist() {
        var dao = getDao();
        assertFalse(dao.changeState("not_exist", State.RUNNING, 42L).join());
    }

    @Test
    public void unableChangeStateByOldSeqNo() {
        var dao = getDao();
        var expected = randomTask();
        assertTrue(dao.add(expected).join());

        assertTrue(dao.changeState(expected.id(), State.RUNNING, 4L).join());
        // previous seqNo lease ownership
        assertTrue(dao.changeState(expected.id(), State.RUNNING, 5L).join());
        // task finished
        assertTrue(dao.changeState(expected.id(), State.SCHEDULED, 5L).join());
        // old seqNo should be ignored now
        assertFalse(dao.changeState(expected.id(), State.RUNNING, 4L).join());
        assertTaskEqual(expected);
    }

    @Test
    public void complete() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(42), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.COMPLETED)
                .setResult(anyAsNumber(42))
                .setStatus(Status.OK)
                .build());
    }

    @Test
    public void unableCompleteByOldSeqNo() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        // start running by other process
        assertTrue(dao.changeState(task.id(), State.RUNNING, 6L).join());
        assertFalse(dao.complete(task.id(), anyAsNumber(42), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.RUNNING)
                .build());
    }

    @Test
    public void unableCompleteAlreadyCompleted() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(1), 5L).join());
        assertFalse(dao.complete(task.id(), anyAsNumber(2), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.COMPLETED)
                .setResult(anyAsNumber(1))
                .setStatus(Status.OK)
                .build());
    }

    @Test
    public void unableChangeStateAlreadyCompleted() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(333), 5L).join());
        assertFalse(dao.changeState(task.id(), State.SCHEDULED, 10L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.COMPLETED)
                .setResult(anyAsNumber(333))
                .setStatus(Status.OK)
                .build());
    }

    @Test
    public void failed() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.failed(task.id(), Status.INTERNAL.withDescription("hi"), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.COMPLETED)
                .setStatus(Status.INTERNAL.withDescription("hi"))
                .build());
    }

    @Test
    public void unableFailByOldSeqNo() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        // start running by other process
        assertTrue(dao.changeState(task.id(), State.RUNNING, 6L).join());
        assertFalse(dao.failed(task.id(), Status.INTERNAL.withDescription("hi"), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.RUNNING)
                .build());
    }

    @Test
    public void unableFailAlreadyCompleted() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(1), 5L).join());
        assertFalse(dao.failed(task.id(), Status.INTERNAL.withDescription("hi"), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.COMPLETED)
                .setResult(anyAsNumber(1))
                .setStatus(Status.OK)
                .build());
    }

    @Test
    public void reschedule() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        long executeAt = task.executeAt() + 10_000;
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.reschedule(task.id(), executeAt, anyAsNumber(123), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.SCHEDULED)
                .setProgress(anyAsNumber(123))
                .setExecuteAt(executeAt)
                .setVersion(task.version() + 1)
                .build());
    }

    @Test
    public void unableRescheduleAlreadyCompleted() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        long executeAt = task.executeAt() + 10_000;
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(123), 5L).join());
        assertFalse(dao.reschedule(task.id(), executeAt, anyAsNumber(123), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.COMPLETED)
                .setResult(anyAsNumber(123))
                .build());
    }

    @Test
    public void unableRescheduleByOldSeqNo() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        // start running by other process
        assertTrue(dao.changeState(task.id(), State.RUNNING, 6L).join());
        assertFalse(dao.reschedule(task.id(), System.currentTimeMillis(), anyAsNumber(42), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.RUNNING)
                .build());
    }

    @Test
    public void rescheduleExternallyWhenScheduled()  {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        long executeAt = task.executeAt() + 10_000;
        assertTrue(dao.changeState(task.id(), State.SCHEDULED, 5L).join());
        assertTrue(dao.rescheduleExternally(task.id(), executeAt, anyAsNumber(123), task.version()).join());

        var expected = task.toBuilder()
            .setState(State.SCHEDULED)
            .setProgress(anyAsNumber(123))
            .setExecuteAt(executeAt)
            .setVersion(task.version() + 1)
            .build();
        assertTaskEqual(expected);

        assertFalse(dao.progress(task.id(), Any.getDefaultInstance(), 5L).join());
        assertTaskEqual(expected);
    }

    @Test
    public void rescheduleExternallyWhenRunning()  {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        long executeAt = task.executeAt() + 10_000;
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.rescheduleExternally(task.id(), executeAt, anyAsNumber(123), task.version()).join());

        var expected = task.toBuilder()
            .setState(State.RUNNING)
            .setProgress(anyAsNumber(123))
            .setExecuteAt(executeAt)
            .setVersion(task.version() + 1)
            .build();
        assertTaskEqual(expected);

        assertFalse(dao.progress(task.id(), Any.getDefaultInstance(), 5L).join());
        assertTaskEqual(expected);
    }

    @Test
    public void unableRescheduleExternallyWhenVersionDiffers() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        long executeAt = task.executeAt() + 10_000;
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.rescheduleExternally(task.id(), executeAt, anyAsNumber(123), task.version()).join());

        var expected = task.toBuilder()
            .setState(State.RUNNING)
            .setProgress(anyAsNumber(123))
            .setExecuteAt(executeAt)
            .setVersion(task.version() + 1)
            .build();
        assertTaskEqual(expected);

        assertFalse(dao.rescheduleExternally(task.id(), executeAt + 1234, anyAsNumber(42), task.version()).join());
    }

    @Test
    public void unableRescheduleExternallyWhenAlreadyCompleted() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        long executeAt = task.executeAt() + 10_000;
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.progress(task.id(), anyAsNumber(1337), 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(42), 5L).join());
        assertFalse(dao.rescheduleExternally(task.id(), executeAt, anyAsNumber(123), task.version()).join());

        assertTaskEqual(task.toBuilder()
                            .setState(State.COMPLETED)
                            .setProgress(anyAsNumber(1337))
                            .setResult(anyAsNumber(42))
                            .setVersion(task.version() + 1)
                            .build());
    }

    @Test
    public void unableRescheduleExternallyWhenUnknown() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        long executeAt = task.executeAt() + 10_000;
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertFalse(dao.rescheduleExternally("unknown-id", executeAt, anyAsNumber(123), task.version()).join());

        assertTaskEqual(task.toBuilder()
                            .setState(State.RUNNING)
                            .build());
    }

    @Test
    public void progress() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.progress(task.id(), anyAsNumber(1), 5L).join());
        assertTrue(dao.progress(task.id(), anyAsNumber(2), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.RUNNING)
                .setProgress(anyAsNumber(2))
                .setVersion(task.version() + 2)
                .build());
    }

    @Test
    public void unableProgressAlreadyCompleted() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(123), 5L).join());
        assertFalse(dao.progress(task.id(), anyAsNumber(333), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.COMPLETED)
                .setResult(anyAsNumber(123))
                .build());
    }

    @Test
    public void unableProgressByOldSeqNo() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        // start running by other process
        assertTrue(dao.changeState(task.id(), State.RUNNING, 6L).join());
        assertFalse(dao.progress(task.id(), anyAsNumber(42), 5L).join());

        assertTaskEqual(task.toBuilder()
                .setState(State.RUNNING)
                .build());
    }

    @Test
    public void scheduledEmpty() {
        assertScheduled(System.currentTimeMillis(), List.of());
    }

    @Test
    public void scheduledNotTimeToExec() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());
        assertScheduled(task.executeAt() - 100, List.of());
    }

    @Test
    public void scheduledOne() {
        var dao = getDao();
        var task = randomTask();
        assertTrue(dao.add(task).join());
        assertScheduled(task.executeAt() + 100, List.of(task));
    }

    @Test
    public void scheduledLimited() {
        var dao = getDao();
        List<Task> tasks = new ArrayList<>();
        AsyncActorBody body = () -> {
            if (tasks.size() == 1004) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var task = randomTask();
            tasks.add(task);
            return dao.add(task);
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, MoreExecutors.directExecutor(), 10);
        runner.start().join();

        tasks.sort(Comparator.comparingLong(Task::executeAt));

        var last = tasks.get(tasks.size() - 1);
        var actual = dao.listScheduled(last.executeAt(), 3).join();
        assertEquals(3, actual.size());
    }

    @Test
    public void scheduledExcludeCompleted() {
        var dao = getDao();
        var task = randomTask();

        assertTrue(dao.add(task).join());
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.complete(task.id(), anyAsNumber(123), 5L).join());

        assertScheduled(task.executeAt() + 100, List.of());
    }

    @Test
    public void scheduledExcludeFailed() {
        var dao = getDao();
        var task = randomTask();

        assertTrue(dao.add(task).join());
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.failed(task.id(), Status.ABORTED, 5L).join());

        assertScheduled(task.executeAt() + 100, List.of());
    }

    @Test
    public void scheduledIncludeRunning() {
        var dao = getDao();
        var task = randomTask();

        assertTrue(dao.add(task).join());
        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());

        assertScheduled(task.executeAt() + 100, List.of(task));
    }

    @Test
    public void scheduledReschedule() {
        var dao = getDao();
        var task = randomTask();

        assertTrue(dao.add(task).join());
        assertScheduled(task.executeAt(), List.of(task));

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.reschedule(task.id(), task.executeAt() + 30_000, anyAsNumber(3), 5L).join());

        assertScheduled(task.executeAt(), List.of());
        assertScheduled(task.executeAt() + 30_000, List.of(task.toBuilder().setExecuteAt(task.executeAt() + 30_000).build()));
    }

    @Test
    public void listTaskEmpty() {
        var dao = getDao();
        var result = new ArrayList<Task>();
        dao.list(result::add).join();
        assertEquals(new ArrayList<Task>(), result);
    }

    @Test
    public void listTasks() {
        var dao = getDao();
        var expected = new ArrayList<Task>();
        for (int index = 0; index < 3; index++) {
            var task = randomTask();
            expected.add(task);
            assertTrue(dao.add(task).join());
        }

        var result = new ArrayList<Task>();
        dao.list(result::add).join();

        expected.sort(Comparator.comparing(Task::id));
        result.sort(Comparator.comparing(Task::id));

        assertEquals(expected, result);
    }

    @Test
    public void rescheduleOnSameTime() {
        var dao = getDao();
        var task = randomTask();

        assertTrue(dao.add(task).join());
        assertScheduled(task.executeAt(), List.of(task));

        assertTrue(dao.changeState(task.id(), State.RUNNING, 5L).join());
        assertTrue(dao.reschedule(task.id(), task.executeAt(), anyAsNumber(3), 5L).join());

        assertScheduled(task.executeAt(), List.of(task));
    }

    @Test
    public void unableProgressForNotRunningTask() {
        var dao = getDao();
        var task = randomTask();

        assertTrue(dao.add(task).join());
        assertScheduled(task.executeAt(), List.of(task));

        assertFalse(dao.progress(task.id(), anyAsNumber(1), 5L).join());
        assertEquals(task, dao.get(task.id()).join().orElseThrow());
    }

    private void assertScheduled(long now, List<Task> tasks) {
        var dao = getDao();
        var actual = Set.copyOf(dao.listScheduled(now, Integer.MAX_VALUE).join());
        var expected = tasks.stream()
                .map(task -> new ScheduledTask(task.executeAt(), task.id(), task.type(), task.params()))
                .collect(Collectors.toSet());
        assertEquals(expected, actual);
    }

    private void assertTaskEqual(Task expected) {
        var dao = getDao();
        var opt = dao.get(expected.id()).join();
        assertTrue(opt.isPresent());
        var actual = opt.get();
        assertEquals(clearStatus(expected), clearStatus(actual));
        assertStatusEqual(expected.status(), actual.status());
    }

    private static void assertStatusEqual(Status expected, Status actual) {
        assertEquals(actual.toString(), expected.getCode(), actual.getCode());
        assertEquals(actual.toString(), Strings.nullToEmpty(expected.getDescription()), Strings.nullToEmpty(actual.getDescription()));
    }

    private static Task clearStatus(Task task) {
        return task.toBuilder()
                .setStatus(Status.OK)
                .build();
    }
}
