package ru.yandex.solomon.scheduler.handlers;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.Descriptors.Descriptor;
import com.google.protobuf.Int32Value;
import com.google.protobuf.StringValue;

import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.util.Proto;

import static java.util.concurrent.CompletableFuture.delayedExecutor;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphanumeric;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public class CooperativeTaskHandler implements TaskHandler {

    public static final String INTERRUPTED = "interrupted";
    public static final String CANCELED = "canceled";
    public static final String DONE = "done";

    @Override
    public String type() {
        return "cooperative";
    }

    @Override
    @Nullable
    public Permit acquire(String id, Any params) {
        return Permits.acquire();
    }

    @Override
    public void execute(ExecutionContext context) {
        var initialProgress = Proto.unpack(context.task().progress(), StringValue.getDefaultInstance());
        if (INTERRUPTED.equals(initialProgress.getValue())) {
            context.complete(Proto.pack(StringValue.of(CANCELED)));
        } else {
            var workToDo = Proto.unpack(context.task().params(), Int32Value.getDefaultInstance()).getValue();
            workRoutine(context, workToDo);
        }
    }

    @Override
    public List<Descriptor> descriptors() {
        return List.of(StringValue.getDescriptor(), Int32Value.getDescriptor());
    }

    private CompletableFuture<?> workRoutine(ExecutionContext context, int workToDo) {
        if (workToDo == 0) {
            return context.complete(Proto.pack(StringValue.of(DONE)));
        }

        return context.progress(Proto.pack(StringValue.of(randomAlphanumeric(4))))
            .thenComposeAsync(
                i -> workRoutine(context, workToDo - 1),
                delayedExecutor(1, TimeUnit.MILLISECONDS));
    }

}
