package ru.yandex.solomon.scheduler.handlers;

import java.util.List;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import com.google.protobuf.Any;
import com.google.protobuf.Int32Value;
import com.google.protobuf.StringValue;

import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.util.Proto;

/**
 * @author Vladimir Gordiychuk
 */
public class Tasks {

    public static List<TaskHandler> handlers() {
        return List.of(
                new ContextTaskHandler(),
                new FutureTaskHandler(),
                new NumberTaskHandler(),
                new StringTaskHandler(),
                new CooperativeTaskHandler()
        );
    }

    public static Task futureTask() {
        return futureTask(0);
    }

    public static Task futureTask(long executeAt) {
        return task()
                .setType("future")
                .setExecuteAt(executeAt)
                .build();
    }

    public static Task contextTask() {
        return contextTask(0);
    }

    public static Task contextTask(long executeAt) {
        return task()
                .setType("context")
                .setExecuteAt(executeAt)
                .build();
    }

    public static Task incTask() {
        return incTask(0);
    }

    public static Task randomTask() {
        int type = ThreadLocalRandom.current().nextInt(6);
        var now = System.currentTimeMillis();
        long executeAt = ThreadLocalRandom.current().nextLong(now, now + TimeUnit.DAYS.toMillis(100));
        return switch (type) {
            case 0 -> Tasks.getTask(executeAt);
            case 1 -> Tasks.incTask(executeAt);
            case 2 -> Tasks.decTask(executeAt);
            case 3 -> Tasks.futureTask(executeAt);
            case 4 -> Tasks.contextTask(executeAt);
            default -> Tasks.concatTask("value=" + ThreadLocalRandom.current().nextLong(), executeAt);
        };
    }

    public static Task incTask(long executeAt) {
        return task()
                .setType("number")
                .setParams(Proto.pack(StringValue.of("inc")))
                .setExecuteAt(executeAt)
                .build();
    }

    public static Task decTask() {
        return decTask(0);
    }

    public static Task decTask(long executeAt) {
        return task()
                .setType("number")
                .setParams(Proto.pack(StringValue.of("dec")))
                .setExecuteAt(executeAt)
                .build();
    }

    public static Task getTask() {
        return getTask(0);
    }

    public static Task getTask(long executeAt) {
        return task()
                .setType("number")
                .setParams(Proto.pack(StringValue.of("get")))
                .setExecuteAt(executeAt)
                .build();
    }

    public static Task concatTask(String string) {
        return concatTask(string, 0);
    }

    public static Task concatTask(String string, long executeAt) {
        return task()
                .setType("string")
                .setParams(Proto.pack(StringValue.of(string)))
                .setExecuteAt(executeAt)
                .build();
    }

    public static Task cooperativeTask() {
        return cooperativeTask(0);
    }

    public static Task cooperativeTask(long executeAt) {
        return cooperativeTask(executeAt, Integer.MAX_VALUE);
    }

    public static Task cooperativeTask(long executeAt, int workToDo) {
        return task()
            .setType("cooperative")
            .setParams(Proto.pack(Int32Value.of(workToDo)))
            .setExecuteAt(executeAt)
            .build();
    }

    public static Any anyAsNumber(int num) {
        return Any.pack(Int32Value.of(num));
    }

    public static Any anyAsNumber() {
        return anyAsNumber(ThreadLocalRandom.current().nextInt());
    }

    private static Task.Builder task() {
        return Task.newBuilder()
                .setId(UUID.randomUUID().toString())
                .setVersion(ThreadLocalRandom.current().nextInt(0,  3));
    }
}
