package ru.yandex.solomon.exception.handlers;

import java.net.ConnectException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.springframework.beans.TypeMismatchException;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.validation.BindException;
import org.springframework.web.HttpMediaTypeNotAcceptableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.multipart.support.MissingServletRequestPartException;
import org.springframework.web.server.ResponseStatusException;

import ru.yandex.solomon.alert.gateway.endpoint.AlertServiceException;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.auth.openid.OpenIdCookies;
import ru.yandex.solomon.core.conf.UnknownShardException;
import ru.yandex.solomon.core.exceptions.NotOwnerException;
import ru.yandex.solomon.expression.exceptions.SelException;
import ru.yandex.solomon.labels.query.SelectorsException;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metrics.client.MetabaseClientException;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.metrics.client.exceptions.DataClientException;
import ru.yandex.solomon.util.UnknownShardLocation;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.stockpile.api.EStockpileStatusCode;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public final class CommonApiExceptionHandler {
    public static ExceptionData tryHandleSolomonException(Throwable cause, @Nullable String requestId, boolean isInternalApi) {
        final Status.Code grpcStatus;
        HttpStatus httpStatus = null;
        String message = "";
        Map<String, Object> attrs = Map.of();
        Map<String, String> headers = Map.of();

        // Solomon specific exceptions
        if (cause instanceof AuthenticationException e) {
            grpcStatus = Status.Code.UNAUTHENTICATED;
            attrs = new HashMap<>(2);
            attrs.put("redirectTo", e.getRedirectTo());
            if (e.isNeedToResetSessionId()) {
                attrs.put("needToResetSessionId", true);
            }
            if (!e.getSessionState().isEmpty()) {
                headers = Map.of(HttpHeaders.SET_COOKIE, OpenIdCookies.saveSessionState(e.getSessionState()));
            }
        } else if (cause instanceof NotOwnerException e) {
            grpcStatus = Status.Code.PERMISSION_DENIED;
            attrs = Map.of("owner", e.getOwner());
        } else if (cause instanceof MetabaseClientException e) {
            MetabaseStatus status = e.getStatus();
            grpcStatus = mapToGrpcStatus(status.getCode());
            message = "Metabase: " + status.getDescription();
        } else if (cause instanceof StockpileClientException e) {
            StockpileStatus status = e.getStatus();
            grpcStatus = mapToGrpcStatus(status.getCode());
            message = "Stockpile: " + status.getDescription();
        } else if (cause instanceof UnknownShardException) {
            grpcStatus = Status.Code.NOT_FOUND;
        } else if (cause instanceof UnknownShardLocation) {
            grpcStatus = Status.Code.UNAVAILABLE;
        } else if (cause instanceof AlertServiceException e) {
            grpcStatus = mapToGrpcStatus(e.getStatusCode());
            attrs = Map.of("type", e.getStatusCode().name());
        } else if (cause instanceof StatusRuntimeException e) {
            Status status = e.getStatus();
            grpcStatus = status.getCode();
            message = status.getDescription();
            attrs = Map.of("type", status.getCode().name());
        } else if (cause instanceof SelException e) {
            grpcStatus = Status.Code.INVALID_ARGUMENT;
            attrs = Map.of(
                    "type", e.getType(),
                    "details", e.getDetails()
            );
        } else if (cause instanceof DataClientException e) {
            grpcStatus = Status.Code.INVALID_ARGUMENT;
            attrs = Map.of(
                    "type", e.getType(),
                    "details", e.getDetails()
            );
        } else if (cause instanceof SelectorsException) {
            grpcStatus = Status.Code.INVALID_ARGUMENT;
            message = Throwables.getRootCause(cause).getMessage();
        } else {
            // Response status specific exceptions (except AuthorizationException, NotOwnerException, etc)
            ResponseStatus responseStatus = cause.getClass().getAnnotation(ResponseStatus.class);
            if (responseStatus != null) {
                httpStatus = responseStatus.value();
                grpcStatus = mapToGrpcStatus(httpStatus);
            }
            // Spring specific exceptions
            else if (cause instanceof HttpRequestMethodNotSupportedException) {
                grpcStatus = Status.Code.UNIMPLEMENTED;
                httpStatus = HttpStatus.METHOD_NOT_ALLOWED;
                String[] supportedMethods = ((HttpRequestMethodNotSupportedException) cause).getSupportedMethods();
                if (supportedMethods != null) {
                    headers = Map.of(HttpHeaders.ALLOW, StringUtils.arrayToDelimitedString(supportedMethods, ", "));
                }
            } else if (cause instanceof HttpMediaTypeNotSupportedException) {
                grpcStatus = Status.Code.UNIMPLEMENTED;
                httpStatus = HttpStatus.UNSUPPORTED_MEDIA_TYPE;
                List<MediaType> mediaTypes = ((HttpMediaTypeNotSupportedException) cause).getSupportedMediaTypes();
                if (!CollectionUtils.isEmpty(mediaTypes)) {
                    headers = Map.of(HttpHeaders.ACCEPT, MediaType.toString(mediaTypes));
                }
            } else if (cause instanceof HttpMediaTypeNotAcceptableException) {
                grpcStatus = Status.Code.UNIMPLEMENTED;
                httpStatus = HttpStatus.NOT_ACCEPTABLE;
            } else if (cause instanceof ServletRequestBindingException
                    || cause instanceof TypeMismatchException
                    || cause instanceof HttpMessageNotReadableException
                    || cause instanceof MethodArgumentNotValidException
                    || cause instanceof MissingServletRequestPartException
                    || cause instanceof BindException
                    || cause instanceof IllegalArgumentException) {
                grpcStatus = Status.Code.INVALID_ARGUMENT;
            } else if (cause instanceof ResponseStatusException e) {
                httpStatus = e.getStatus();
                grpcStatus = mapToGrpcStatus(httpStatus);
            }
            // Java specific exceptions
            else if (cause instanceof ConnectException) {
                grpcStatus = Status.Code.UNAVAILABLE;
            } else if (cause instanceof IllegalStateException &&
                    "In a WebFlux application, form data is accessed via ServerWebExchange.getFormData().".equals(cause.getMessage())) {
                //Exception text is from org.springframework.web.reactive.result.method.annotation.AbstractMessageReaderArgumentResolver
                grpcStatus = Status.Code.UNIMPLEMENTED;
                httpStatus = HttpStatus.UNSUPPORTED_MEDIA_TYPE;
                message = "Doesn't support application/x-www-form-urlencoded content-type";
            } else {
                // Uncaught exception (UUID generation for backward comparatively)
                grpcStatus = Status.Code.INTERNAL;

                final String exceptionId;
                if (StringUtils.isEmpty(requestId)) {
                    exceptionId = "generated ID: " + UUID.randomUUID();
                } else {
                    exceptionId = "x-request-id: " + requestId;
                }

                if (isInternalApi) {
                    String errorMessage = cause.getMessage();
                    if (StringUtils.isEmpty(errorMessage)) {
                        errorMessage = "uncaught error";
                    }
                    message = "Internal server error, " + exceptionId + ", message: " + errorMessage;
                } else {
                    message = "Internal server error, " + exceptionId;
                }
            }
        }

        // HTTP status is null and calculated from gRPC status by default.
        // In some cases (HTTP-specific or Spring exceptions) gRPC and HTTP statuses may be set independently
        // of each other.
        if (httpStatus == null) {
            httpStatus = mapToHttpStatus(grpcStatus);
        }

        if (StringUtils.isEmpty(message)) {
            message = Nullables.orEmpty(cause.getMessage());
        }

        // Hide useless internal details
        if (message.contains("Required request body is missing")) {
            message = "Required request body is missing";
        }

        // Hide another useless internal details
        if (message.contains("Request body is missing:")) {
            message = "Request body is missing";
        }

        // Message cannot be empty
        if (message.isEmpty()) {
            message = "Uncaught error";
        }

        return new ExceptionData(
                grpcStatus,
                httpStatus,
                message,
                headers,
                attrs);
    }

    private static Status.Code mapToGrpcStatus(EStockpileStatusCode code) {
        return switch (code) {
            case OK -> Status.Code.OK;
            case METRIC_ALREADY_EXISTS -> Status.Code.ALREADY_EXISTS;
            case DEADLINE_EXCEEDED -> Status.Code.DEADLINE_EXCEEDED;
            case SHARD_NOT_READY, NODE_UNAVAILABLE -> Status.Code.UNAVAILABLE;
            case RESOURCE_EXHAUSTED -> Status.Code.RESOURCE_EXHAUSTED;
            default -> Status.Code.INTERNAL;
        };
    }

    private static Status.Code mapToGrpcStatus(EMetabaseStatusCode code) {
        return switch (code) {
            case OK -> Status.Code.OK;
            case DEADLINE_EXCEEDED -> Status.Code.DEADLINE_EXCEEDED;
            case SHARD_WRITE_ONLY, INVALID_REQUEST -> Status.Code.FAILED_PRECONDITION;
            case RESOURCE_EXHAUSTED -> Status.Code.RESOURCE_EXHAUSTED;
            case NOT_FOUND, SHARD_NOT_FOUND -> Status.Code.NOT_FOUND;
            case NODE_UNAVAILABLE, SHARD_NOT_READY -> Status.Code.UNAVAILABLE;
            case DUPLICATE -> Status.Code.ALREADY_EXISTS;
            default -> Status.Code.INTERNAL;
        };
    }

    private static Status.Code mapToGrpcStatus(ERequestStatusCode status) {
        return switch (status) {
            case OK -> Status.Code.OK;
            case NOT_FOUND -> Status.Code.NOT_FOUND;
            case INVALID_REQUEST -> Status.Code.INVALID_ARGUMENT;
            case NOT_AUTHORIZED -> Status.Code.UNAUTHENTICATED;
            case NODE_UNAVAILABLE, SHARD_NOT_INITIALIZED -> Status.Code.UNAVAILABLE;
            case DEADLINE_EXCEEDED -> Status.Code.DEADLINE_EXCEEDED;
            case RESOURCE_EXHAUSTED -> Status.Code.RESOURCE_EXHAUSTED;
            case CONCURRENT_MODIFICATION -> Status.Code.ALREADY_EXISTS;
            default -> Status.Code.INTERNAL;
        };
    }

    private static HttpStatus mapToHttpStatus(Status.Code status) {
        return switch (status) {
            case OK -> HttpStatus.OK;
            case NOT_FOUND -> HttpStatus.NOT_FOUND;
            case UNAVAILABLE -> HttpStatus.SERVICE_UNAVAILABLE;
            case INVALID_ARGUMENT, FAILED_PRECONDITION, OUT_OF_RANGE -> HttpStatus.BAD_REQUEST;
            case ALREADY_EXISTS, ABORTED -> HttpStatus.CONFLICT;
            case PERMISSION_DENIED -> HttpStatus.FORBIDDEN;
            case UNAUTHENTICATED -> HttpStatus.UNAUTHORIZED;
            case RESOURCE_EXHAUSTED -> HttpStatus.TOO_MANY_REQUESTS;
            case CANCELLED, DEADLINE_EXCEEDED -> HttpStatus.GATEWAY_TIMEOUT;
            case UNIMPLEMENTED -> HttpStatus.NOT_IMPLEMENTED;
            default -> HttpStatus.INTERNAL_SERVER_ERROR;
        };
    }

    public static Status.Code mapToGrpcStatus(HttpStatus status) {
        return switch (status) {
            case OK -> Status.Code.OK;
            case BAD_REQUEST -> Status.Code.INVALID_ARGUMENT;
            case UNAUTHORIZED -> Status.Code.UNAUTHENTICATED;
            case FORBIDDEN -> Status.Code.PERMISSION_DENIED;
            case NOT_FOUND -> Status.Code.NOT_FOUND;
            case CONFLICT -> Status.Code.ALREADY_EXISTS;
            case TOO_MANY_REQUESTS -> Status.Code.RESOURCE_EXHAUSTED;
            case NOT_IMPLEMENTED -> Status.Code.DEADLINE_EXCEEDED;
            case SERVICE_UNAVAILABLE -> Status.Code.UNAVAILABLE;
            default -> Status.Code.INTERNAL;
        };
    }
}
