package ru.yandex.solomon.exception.handlers;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.core.io.buffer.PooledDataBuffer;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.util.StringUtils;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebExceptionHandler;
import reactor.core.publisher.Mono;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.http.HttpUtils;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class HttpApiExceptionHandler implements WebExceptionHandler {

    private static final Logger logger = LoggerFactory.getLogger(HttpApiExceptionHandler.class);

    private final ObjectMapper objectMapper;

    @VisibleForTesting
    public HttpApiExceptionHandler(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    @Nonnull
    @Override
    public Mono<Void> handle(@Nonnull ServerWebExchange exchange, @Nonnull Throwable t) {
        /* Exception preprocessing part */
        ServerHttpRequest request = exchange.getRequest();
        String requestId = HttpUtils.requestId(request);
        if (!StringUtils.isEmpty(requestId)) {
            logger.info("Resolve exception by x-request-id identifier: " + requestId);
        }
        if (exchange.getResponse().isCommitted()) {
            if (!ExceptionUtils.isBrokenPipe(t)) {
                // mute broken pipe exceptions
                logger.warn("cannot send response on exception", t);
            }
            return Mono.empty();
        }

        String requestURI = request.getPath().value();
        boolean isInternalApi = requestURI.startsWith("/api")
                || requestURI.startsWith("/rest")
                || requestURI.startsWith("/data-api")
                || requestURI.startsWith("/staffOnly")
                || requestURI.startsWith("/balancer");

        boolean isApiV3 = requestURI.startsWith("/monitoring/v3")
                || requestURI.startsWith("/api/v3");

        Throwable cause = CompletableFutures.unwrapCompletionException(t);

        /* Exception mapping part */

        ExceptionData exceptionData = CommonApiExceptionHandler.tryHandleSolomonException(cause, requestId, isInternalApi);

        /* Exception postprocessing part */

        String message = exceptionData.getMessage();
        if (exceptionData.getHttpStatus() == HttpStatus.INTERNAL_SERVER_ERROR) {
            logger.error(format(request) + " API internal server exception: " + message, cause);
        }

        return send(exchange, exceptionData, isApiV3);
    }

    private Mono<Void> send(
            ServerWebExchange exchange,
            ExceptionData exceptionData,
            boolean isApiV3) {
        final Map<String, Object> error;
        int httpCode = exceptionData.getHttpStatus().value();
        if (isApiV3) {
            // See https://cloud.yandex.ru/docs/api-design-guide/concepts/errors#http-mapping
            error = new HashMap<>(2);
            int grpcCode = exceptionData.getGrpcStatus().value();
            error.put("code", grpcCode);
            error.put("message", exceptionData.getMessage());
        } else {
            // See https://solomon.yandex-team.ru/docs/api-ref/rest#error
            error = new HashMap<>(exceptionData.getAttrs());
            error.put("code", httpCode);
            error.put("message", exceptionData.getMessage());
        }

        ServerHttpResponse response = exchange.getResponse();
        response.setRawStatusCode(httpCode);
        response.getHeaders().setAll(exceptionData.getHeaders());
        response.getHeaders().setContentType(MediaType.APPLICATION_JSON);

        DataBufferFactory bufferFactory = response.bufferFactory();
        DataBuffer body;
        try {
            body = bufferFactory.wrap(objectMapper.writeValueAsBytes(error));
        } catch (JsonProcessingException e) {
            body = bufferFactory.wrap("{}".getBytes(Charsets.US_ASCII));
        }

        return response.writeWith(Mono.just(body))
                .doOnDiscard(PooledDataBuffer.class, DataBufferUtils::release);
    }

    private static String format(ServerHttpRequest request) {
        String remoteAddr = HttpUtils.realOrRemoteIp(request);
        return remoteAddr + ' ' + requestId(request) + ' ' + request.getMethod() + ' ' + HttpUtils.requestUri(request);
    }

    private static String requestId(ServerHttpRequest request) {
        String requestId = HttpUtils.requestId(request);
        return Strings.isNullOrEmpty(requestId) ? "none" : requestId;
    }

}
