package ru.yandex.solomon.expression;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.expression.expr.func.SelFunc;
import ru.yandex.solomon.expression.expr.func.SelOp;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.version.SelVersion;

/**
 * @author Vladimir Gordiychuk
 */
public class ExpressionMetrics implements MetricSupplier {
    public static final ExpressionMetrics I = new ExpressionMetrics();

    private final MetricRegistry registry = new MetricRegistry();

    private final ConcurrentMap<FnKey, FnMetrics> functions = new ConcurrentHashMap<>();
    private final ConcurrentMap<OpKey, OpMetrics> operators = new ConcurrentHashMap<>();

    private final EnumMap<SelVersion, Histogram> compileTimeMs = new EnumMap<>(Arrays.stream(SelVersion.values())
            .collect(Collectors.toMap(Function.identity(), version ->
                    registry.histogramRate("expression.compile.elapsedMicroseconds", Labels.of("version", version.name()),
                            Histograms.exponential(20, 2, 128)))));
    private final EnumMap<SelVersion, Histogram> prepareTimeMs = new EnumMap<>(Arrays.stream(SelVersion.values())
            .collect(Collectors.toMap(Function.identity(), version ->
                    registry.histogramRate("expression.prepare.elapsedMicroseconds", Labels.of("version", version.name()),
                            Histograms.exponential(20, 2, 128)))));
    private final EnumMap<SelVersion, Histogram> evalTimeMs = new EnumMap<>(Arrays.stream(SelVersion.values())
            .collect(Collectors.toMap(Function.identity(), version ->
                    registry.histogramRate("expression.eval.elapsedMicroseconds", Labels.of("version", version.name()),
                            Histograms.exponential(20, 2, 128)))));

    private ExpressionMetrics() {
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    public void compiled(SelVersion version, long elapsedNanos) {
        compileTimeMs.get(version).record(TimeUnit.NANOSECONDS.toMicros(elapsedNanos));
    }

    public void prepared(SelVersion version, long elapsedNanos) {
        prepareTimeMs.get(version).record(TimeUnit.NANOSECONDS.toMicros(elapsedNanos));
    }

    public void evaluated(SelVersion version, long elapsedNanos) {
        evalTimeMs.get(version).record(TimeUnit.NANOSECONDS.toMicros(elapsedNanos));
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }

    public FnMetrics getFnMetrics(SelVersion version, SelFunc fn) {
        FnKey key = new FnKey(version, fn.getName());
        var result = this.functions.get(key);
        if (result != null) {
            return result;
        }

        var labels = Labels.of("function", fn.getName(), "version", version.name());
        result = new FnMetrics(registry.subRegistry(labels));
        this.functions.putIfAbsent(key, result);
        return result;
    }

    public OpMetrics getOpMetrics(SelVersion version, SelOp op) {
        OpKey key = new OpKey(version, op);
        var result = this.operators.get(key);
        if (result != null) {
            return result;
        }

        var labels = Labels.of("operator", op.getName(), "args", formatArgs(op.getArgs()), "version", version.name());
        result = new OpMetrics(registry.subRegistry(labels));
        this.operators.putIfAbsent(key, result);
        return result;
    }

    private String formatArgs(List<SelType> args) {
        if (args.isEmpty()) {
            return "none";
        }

        return args
            .stream()
            .map(SelType::toString)
            .collect(Collectors.joining("/"));
    }

    public static class FnMetrics {
        public final Rate parse;
        public final Rate calls;

        private FnMetrics(MetricRegistry registry) {
            this.parse = registry.rate("expression.functions.parse");
            this.calls = registry.rate("expression.functions.call");
        }
    }

    public static class OpMetrics {
        public final Rate parse;
        public final Rate calls;

        private OpMetrics(MetricRegistry registry) {
            this.parse = registry.rate("expression.operators.parse");
            this.calls = registry.rate("expression.operators.call");
        }
    }

    private static class FnKey {
        private final SelVersion version;
        private final String name;

        private FnKey(SelVersion version, String name) {
            this.version = version;
            this.name = name;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            FnKey fnKey = (FnKey) o;
            return version == fnKey.version &&
                    name.equals(fnKey.name);
        }

        @Override
        public int hashCode() {
            return Objects.hash(version, name);
        }
    }

    private static class OpKey {
        private final SelVersion version;
        private final SelOp op;

        private OpKey(SelVersion version, SelOp op) {
            this.version = version;
            this.op = op;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            OpKey opKey = (OpKey) o;
            return version == opKey.version &&
                    Objects.equals(op, opKey.op);
        }

        @Override
        public int hashCode() {
            return Objects.hash(version, op);
        }
    }
}
