package ru.yandex.solomon.expression;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class LineCountingBuffer {
    public static int EOF = -1;

    private final String data;
    private int offset;
    private final Position currentPosition;
    private final Position prevToCurrentPosition;

    private LineCountingBuffer(String data, int offset, Position currentPosition, Position prevToCurrentPosition) {
        this.data = data;
        this.offset = offset;
        this.currentPosition = currentPosition;
        this.prevToCurrentPosition = prevToCurrentPosition;
    }

    public LineCountingBuffer(String data) {
        this(data, 0, new Position(), new Position());
    }

    public LineCountingBuffer copy() {
        return new LineCountingBuffer(data, offset, currentPosition.copy(), prevToCurrentPosition.copy());
    }

    /**
     * Scans the next char in the buffer
     * @return next char or EOF if the end of buffer is reached
     */
    public int readChar() {
        int c = peekChar();
        if (c != EOF) {
            advance((char)c);
        }
        return c;
    }

    /**
     * Same as nextChar() but does not advance internal pointers
     */
    public int peekChar() {
        if (offset < data.length()) {
            return data.charAt(offset);
        }
        return EOF;
    }

    /**
     * @return Copy of the position in the source. That is line:column
     */
    public Position getCurrentPosition() {
        return currentPosition.copy();
    }

    /**
     * @return Copy of the position immediately before current in the source
     */
    public Position getPrevToCurrentPosition() {
        return prevToCurrentPosition.copy();
    }

    public void advance(char c) {
        prevToCurrentPosition.set(currentPosition);
        currentPosition.advanceOnChar(c);
        offset++;
    }

    public void advance(String s) {
        if (s.isEmpty()) {
            return;
        }
        // TODO: optimize substring out
        int upToLastLen = s.length() - 1;
        String ss = s.substring(0, upToLastLen);
        char lastChar = s.charAt(upToLastLen);
        currentPosition.advanceOnString(ss);
        offset += upToLastLen;
        advance(lastChar);
    }

    public String readRemaining() {
        String remaining = data.substring(offset);
        advance(remaining);
        return remaining;
    }

    public String getData() {
        return data;
    }

    public int getOffset() {
        return offset;
    }

    @Override
    public String toString() {
        return "LineCountingBuffer{" +
            "data='" + data + '\'' +
            ", offset=" + offset +
            ", currentPosition=" + currentPosition +
            ", prevToCurrentPosition=" + prevToCurrentPosition +
            '}';
    }
}
