package ru.yandex.solomon.expression;

import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.model.point.DataPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.model.timeseries.aggregation.TimeseriesSummary;

import static ru.yandex.solomon.model.point.column.StockpileColumns.ensureColumnSetValidIfNotEmpty;


/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class NamedGraphData implements MemMeasurable {
    private final String alias;
    private final MetricType metricType;
    private final String metricName;
    private final Labels labels;
    private final ru.yandex.solomon.model.protobuf.MetricType dataType;
    private final AggrGraphDataArrayListOrView graphData;
    @Nullable
    private final TimeseriesSummary summary;

    public NamedGraphData(
        String alias,
        MetricType metricType,
        String metricName,
        Labels labels,
        ru.yandex.solomon.model.protobuf.MetricType dataType,
        AggrGraphDataArrayListOrView graphData)
    {
        this.alias = alias;
        this.metricType = metricType;
        this.metricName = metricName;
        this.labels = labels;
        this.dataType = dataType;
        this.graphData = graphData;
        this.summary = null;

        ensureColumnSetValidIfNotEmpty(dataType, graphData.columnSetMask());
    }

    @VisibleForTesting
    @Deprecated
    public NamedGraphData(
            String alias,
            MetricType metricType,
            String metricName,
            Labels labels,
            AggrGraphDataArrayList graphData)
    {
        this.alias = alias;
        this.metricType = metricType;
        this.metricName = metricName;
        this.labels = labels;
        this.dataType = StockpileColumns.typeByMask(graphData.columnSetMask());
        this.graphData = graphData;
        this.summary = null;

        ensureColumnSetValidIfNotEmpty(dataType, graphData.columnSetMask());
    }

    private NamedGraphData(Builder builder) {
        alias = builder.alias;
        metricType = builder.metricType;
        metricName = builder.metricName;
        labels = builder.labels;
        dataType = Objects.requireNonNull(builder.dataType, "");
        graphData = Objects.requireNonNull(builder.graphData, "");
        summary = builder.summary;

        ensureColumnSetValidIfNotEmpty(dataType, graphData.columnSetMask());
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Builder toBuilder() {
        Builder builder = new Builder();
        builder.alias = this.alias;
        builder.metricType = this.metricType;
        builder.metricName = this.metricName;
        builder.labels = this.labels;
        builder.dataType = this.dataType;
        builder.graphData = this.graphData;
        builder.summary = this.summary;
        return builder;
    }

    // For test support only
    public static NamedGraphData of(GraphData graphData) {
        return NamedGraphData.newBuilder()
            .setGraphData(graphData)
            .build();
    }

    @VisibleForTesting
    @Deprecated
    public static NamedGraphData of(AggrGraphDataArrayListOrView source) {
        return of(StockpileColumns.typeByMask(source.columnSetMask()), source);
    }

    public static NamedGraphData of(ru.yandex.solomon.model.protobuf.MetricType type, AggrGraphDataArrayListOrView source) {
        return NamedGraphData.newBuilder()
            .setGraphData(type, source)
            .build();
    }

    public static NamedGraphData of(GraphData graphData, String metricName, Labels labels, String alias) {
        return NamedGraphData.newBuilder()
            .setAlias(alias)
            .setMetricName(metricName)
            .setLabels(labels)
            .setGraphData(graphData)
            .build();
    }

    public static NamedGraphData of(Labels labels, DataPoint... points) {
        return of(MetricType.DGAUGE, labels, points);
    }

    public static NamedGraphData of(MetricType type, Labels labels, DataPoint... points) {
        GraphData graphData = GraphData.of(points);
        return NamedGraphData.newBuilder()
            .setType(type)
            .setLabels(labels)
            .setGraphData(graphData)
            .build();
    }

    public static NamedGraphData of(String alias, Labels labels, DataPoint... points) {
        return of(MetricType.DGAUGE, alias, labels, points);
    }

    public static NamedGraphData of(MetricType type, String alias, Labels labels, DataPoint... points) {
        GraphData graphData = GraphData.of(points);
        return NamedGraphData.newBuilder()
            .setType(type)
            .setAlias(alias)
            .setLabels(labels)
            .setGraphData(graphData)
            .build();
    }

    public String getAlias() {
        return alias;
    }

    /**
    * This is type from Metabase, may be inconsistent with actual data
    */
    @Deprecated
    public MetricType getType() {
        return metricType;
    }

    /**
     * This is type derived from data columnSet. Is METRIC_TYPE_UNSPECIFIED if data is empty
     */
    public ru.yandex.solomon.model.protobuf.MetricType getDataType() {
        return dataType;
    }

    public String getMetricName() {
        return metricName;
    }

    public Labels getLabels() {
        return labels;
    }

    @VisibleForTesting
    @Deprecated
    public GraphData getGraphData() {
        return graphData.toGraphDataShort();
    }

    public GraphData getGraphData(ru.yandex.solomon.model.protobuf.MetricType dataType) {
        return graphData.toGraphDataShort(dataType);
    }

    public AggrGraphDataArrayListOrView getAggrGraphDataArrayList() {
        return graphData;
    }

    @Nullable
    public TimeseriesSummary getSummary() {
        return summary;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        NamedGraphData that = (NamedGraphData) o;
        return alias.equals(that.alias) &&
                metricType == that.metricType &&
                metricName.equals(that.metricName) &&
                labels.equals(that.labels) &&
                dataType == that.dataType &&
                graphData.equals(that.graphData) &&
                Objects.equals(summary, that.summary);
    }

    @Override
    public int hashCode() {
        return Objects.hash(alias, metricType, metricName, labels, dataType, graphData, summary);
    }

    @Override
    public String toString() {
        return "NamedGraphData{" +
            "alias='" + alias + '\'' +
            ", type=" + metricType +
            ", metricName='" + metricName + '\'' +
            ", labels=" + labels +
            ", dataType=" + dataType +
            ", graphData=" + graphData +
            ", summary=" + summary +
            '}';
    }

    @Override
    public long memorySizeIncludingSelf() {
        return graphData.memorySizeIncludingSelf();
    }

    public static final class Builder {
        private String alias = "";
        private MetricType metricType = MetricType.DGAUGE;
        private String metricName = "";
        private Labels labels = Labels.empty();
        private ru.yandex.solomon.model.protobuf.MetricType dataType;
        private AggrGraphDataArrayListOrView graphData;
        @Nullable
        private TimeseriesSummary summary;

        private Builder() {
        }

        public Builder setAlias(String alias) {
            this.alias = alias;
            return this;
        }

        public Builder setType(MetricType type) {
            this.metricType = type;
            return this;
        }

        public Builder setMetricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public Builder setLabels(Labels labels) {
            this.labels = labels;
            return this;
        }

        /**
         * The only valid use case of this method is for type preserving transformations, e.g.
         * <pre>
         * ngd.toBuilder()
         *     .setGraphData(transform(ngd.getAggrGraphDataArrayList()))
         * </pre>
         * May be used in tests if deriving type from data columnSet is ok
         */
        @Deprecated
        public Builder setGraphData(AggrGraphDataArrayListOrView graphData) {
            this.dataType = (dataType == null || dataType == ru.yandex.solomon.model.protobuf.MetricType.METRIC_TYPE_UNSPECIFIED)
                    ? StockpileColumns.typeByMask(graphData.columnSetMask()) : dataType;
            this.graphData = graphData;
            return this;
        }

        public Builder setGraphData(ru.yandex.solomon.model.protobuf.MetricType dataType, AggrGraphDataArrayListOrView graphData) {
            ensureColumnSetValidIfNotEmpty(dataType, graphData.columnSetMask());
            this.dataType = dataType;
            this.graphData = graphData;
            return this;
        }

        public Builder setGraphData(GraphData graphData) {
            return this.setGraphData(graphData.toAggrGraphDataArrayList());
        }

        public Builder setSummary(@Nullable TimeseriesSummary summary) {
            this.summary = summary;
            return this;
        }

        public NamedGraphData build() {
            return new NamedGraphData(this);
        }
    }
}
