package ru.yandex.solomon.expression;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ComparisonChain;

/**
 * @author Ivan Tsybulin
 */
public class Position implements Comparable<Position> {
    private int line;
    private int column;
    private int offset;

    public static final Position UNKNOWN = new Position();

    static {
        UNKNOWN.line = -1;
        UNKNOWN.column = -1;
        UNKNOWN.offset = -1;
    }

    Position() {
        this(1, 1, 0);
    }

    @VisibleForTesting
    public Position(int line, int column, int offset) {
        if (line < 1 || column < 1 || offset < 0) {
            throw new IllegalArgumentException("Invalid position: " + line + ":" + column + "@" + offset);
        }
        this.line = line;
        this.column = column;
        this.offset = offset;
    }

    public void set(Position other) {
        this.line = other.line;
        this.column = other.column;
        this.offset = other.offset;
    }

    private void advanceInLine(int columns) {
        column += columns;
        offset += columns;
    }

    private void lineFeedAfterLine(int columns) {
        line++;
        column = 1;
        offset += columns;
    }

    public void advanceOnChar(char c) {
        if (c == '\n') {
            lineFeedAfterLine(1);
        } else {
            advanceInLine(1);
        }
    }

    public void advanceOnString(String str) {
        int p = 0;
        while (p < str.length()) {
            int newline = str.indexOf('\n', p);
            if (newline == -1) {
                // str has no newlines from p till the end
                advanceInLine(str.length() - p);
                break;
            }
            lineFeedAfterLine(newline - p + 1);
            p = newline + 1;
        }
    }

    public Position copy() {
        if (this == UNKNOWN) {
            return UNKNOWN;
        }
        return new Position(line, column, offset);
    }

    public int getLine() {
        return line;
    }

    public int getColumn() {
        return column;
    }

    public int getOffset() {
        return offset;
    }

    @Override
    public String toString() {
        if (this == UNKNOWN) {
            return "unknown";
        }
        return line + ":" + column;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Position position = (Position) o;
        return line == position.line &&
            column == position.column &&
            offset == position.offset;
    }

    @Override
    public int hashCode() {
        int hash = line;
        hash = (hash << 10) ^ column;
        hash = (hash << 10) ^ offset;
        return hash;
    }

    @Override
    public int compareTo(Position other) {
        return ComparisonChain.start()
                .compare(line, other.line)
                .compare(column, other.column)
                .compare(offset, other.offset)
                .result();
    }
}
