package ru.yandex.solomon.expression;

import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class PositionRange {
    public static final PositionRange UNKNOWN = PositionRange.of(Position.UNKNOWN, Position.UNKNOWN);

    private final Position begin;
    private final Position end;

    private PositionRange(Position beginInclusive, Position endInclusive) {
        this.begin = beginInclusive;
        this.end = endInclusive;
    }

    public Position getBegin() {
        return begin;
    }

    public Position getEnd() {
        return end;
    }

    @VisibleForTesting
    public static PositionRange of(int column) {
        return of(column, column);
    }

    @VisibleForTesting
    public static PositionRange of(int columnBegin, int columnEnd) {
        return of(1, columnBegin, columnBegin - 1, columnEnd);
    }

    @VisibleForTesting
    public static PositionRange of(int line, int columnBegin, int offsetBegin, int columnEnd) {
        return new PositionRange(
            new Position(line, columnBegin, offsetBegin),
            new Position(line, columnEnd, offsetBegin + columnEnd - columnBegin));
    }

    @VisibleForTesting
    public static PositionRange of(int lineBegin, int columnBegin, int offsetBegin, int lineEnd, int columnEnd, int offsetEnd) {
        return new PositionRange(
            new Position(lineBegin, columnBegin, offsetBegin),
            new Position(lineEnd, columnEnd, offsetEnd));
    }

    public static PositionRange of(Position beginInclusive, Position endInclusive) {
        return new PositionRange(beginInclusive, endInclusive);
    }

    public static PositionRange convexHull(PositionRange first, PositionRange... rest) {
        return convexHull(first, List.of(rest));
    }

    public static PositionRange convexHull(PositionRange first, List<PositionRange> rest) {
        var begin = first.begin;
        var end = first.begin;
        for (PositionRange other : rest) {
            if (other.begin.compareTo(begin) < 0) {
                begin = other.begin;
            }
            if (other.end.compareTo(end) > 0) {
                end = other.end;
            }
        }
        return new PositionRange(begin.copy(), end.copy());
    }

    @Override
    public String toString() {
        if (begin.equals(end)) {
            return begin.toString();
        }
        if (begin.getLine() == end.getLine()) {
            return begin.getLine() + ":" + begin.getColumn() + "-" + end.getColumn();
        }
        return begin + "-" + end;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PositionRange that = (PositionRange) o;
        return begin.equals(that.begin) &&
            end.equals(that.end);
    }

    @Override
    public int hashCode() {
        return Objects.hash(begin, end);
    }
}
