package ru.yandex.solomon.expression;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang3.StringEscapeUtils;

import ru.yandex.solomon.labels.query.SelectorsFormat;

/**
 * @author Stepan Koltsov
 *
 * This class must not throw exceptions on its own. The reason is because this class knows little about the context of the
 * exception thus its exception often is meaningless to the user.
 */
@ParametersAreNonnullByDefault
final public class SelLexer {

    private static final Pattern DOUBLE_PATTERN = Pattern.compile("(?i)[-+]?[0-9]*\\.?[0-9]+([eE][-+]?[0-9]+)?[KMGTPE]?");
    private static final Pattern DURATION_PATTERN = Pattern.compile("[-+]?(\\d+(ms|[wdhms]))+");
    public static final Pattern IDENT_PATTERN = Pattern.compile("(?i)[a-z_][a-z0-9_]*");
    private static final Pattern IDENT_WITH_DOTS = SelectorsFormat.METRIC_NAME_PATTERN;
    private static final String[] PUNCT_LIST = {
            "!==", "=~", "!~",
            "->",
            "&&", "||",
            ">=", "<=", "==", "!=",
            ">", "<",
            "*", "/", "+", "-",
            "(", ")", ",",
            "!",
            "=", ";",
            "?", ":",
            "{", "}",
            "[", "]"
    };

    public static final Set<String> KEYWORD_SET = Set.of(
            "let",
            "by",
            "return"
    );

    static {
        // Keywords must match ident pattern. Required for consumeKeywordOpt()
        for (String keyword : KEYWORD_SET) {
            assert IDENT_PATTERN.matcher(keyword).matches();
        }
    }

    private final SelScanner scanner;
    private final List<Supplier<ScannedToken>> tokenParsingOrder;

    public SelLexer(String string) {
        this(new SelScanner(string));
    }

    private SelLexer(SelScanner scanner) {
        this.scanner = scanner;
        this.tokenParsingOrder = List.of(
                this::consumeKeywordOpt,
                this::consumePunctOpt,
                this::consumeIdentWithDotsOpt,
                this::consumeDurationOpt,
                this::consumeDoubleOpt,
                this::consumeStringLiteralOpt
            );
    }

    public SelLexer copy() {
        return new SelLexer(scanner.copy());
    }

    public enum TokenType {
        /**
         * @see SelLexer#KEYWORD_SET
         */
        KEYWORD,
        /**
         * @see SelLexer#PUNCT_LIST
         */
        PUNCT,
        /**
         * @see SelLexer#IDENT_PATTERN
         */
        IDENT,
        /**
         * @see SelLexer#IDENT_WITH_DOTS
         */
        IDENT_WITH_DOTS,
        /**
         * @see SelLexer#DOUBLE_PATTERN
         */
        DOUBLE,
        STRING,
        /**
         * @see SelLexer#DOUBLE_PATTERN
         */
        DURATION,

        UNRECOGNIZED,
        EOF,
        ;

        public Token of(String value) {
            return new Token(this, value);
        }

        public ScannedToken of(SelScanner.RangedString rangedString) {
            return new ScannedToken(this, rangedString.getValue(), rangedString.getPos());
        }
    }

    public static final class ScannedToken extends Token {
        private final PositionRange pos;

        private ScannedToken(TokenType tokenType, String value, PositionRange pos) {
            super(tokenType, value);
            this.pos = pos;
        }

        public PositionRange getRange(boolean rangesEnabled) {
            if (rangesEnabled) {
                return pos;
            }
            return PositionRange.UNKNOWN;
        }

        public boolean isPresent() {
            return pos != PositionRange.UNKNOWN;
        }

        public String shortString() {
            return super.toString();
        }

        public boolean matches(Token token) {
            return token.getTokenType() == getTokenType() && token.getValue().equals(getValue());
        }

        public boolean matches(TokenType tokenType) {
            return tokenType == getTokenType();
        }

        @Override
        public String toString() {
            return shortString() + '@' + pos;
        }

        @Override
        @VisibleForTesting
        @Deprecated(since = "For tests only, use matches() of ScannedToken")
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (getClass() != o.getClass()) {
                return false;
            }
            if (!super.equals(o)) {
                return false;
            }
            ScannedToken that = (ScannedToken) o;
            return pos.equals(that.pos);
        }

        @Override
        public int hashCode() {
            return Objects.hash(super.hashCode(), pos);
        }
    }

    public static class Token {
        private final TokenType tokenType;
        private final String value;

        private Token(TokenType tokenType, String value) {
            this.tokenType = tokenType;
            this.value = value;
        }

        public ScannedToken at(PositionRange pos) {
            return new ScannedToken(tokenType, value, pos);
        }

        public TokenType getTokenType() {
            return tokenType;
        }

        public String getValue() {
            return value;
        }

        @Override
        public String toString() {
            // UNRECOGNIZED tokens may have pretty long values
            final int maxValueLen = 30;
            String truncatedValue = (value.length() > maxValueLen) ? (value.substring(0, maxValueLen - 1) + "…") : value;
            return tokenType + "{\"" + StringEscapeUtils.escapeJava(truncatedValue) + "\"}";
        }

        @Override
        @VisibleForTesting
        @Deprecated(since = "For tests only, use matches() of ScannedToken")
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Token token = (Token) o;
            return tokenType == token.tokenType &&
                value.equals(token.value);
        }

        @Override
        public int hashCode() {
            return Objects.hash(tokenType, value);
        }
    }

    public ScannedToken lookaheadToken() {
        return copy().nextToken();
    }

    private void consumeWhitespaceAndComments() {
        while (true) {
            scanner.skipWhitespace();
            if (scanner.consume("//") != SelScanner.FAILED) {
                scanner.skipRestOfLine();
                continue;
            }
            break;
        }
    }

    /**
    * @return Token of KEYWORD type. If token was not consumed then SelLexer.Token#isPresent() returns false
    * */
    private ScannedToken consumeKeywordOpt() {
        SelScanner.RangedString rangedString = scanner.copy().consumePattern(IDENT_PATTERN);
        if (KEYWORD_SET.contains(rangedString.getValue())) {
            return TokenType.KEYWORD.of(scanner.consume(rangedString.getValue()));
        }
        return TokenType.KEYWORD.of(SelScanner.FAILED);
    }

    /**
     * @see SelLexer#consumeKeywordOpt()
     */
    private ScannedToken consumePunctOpt() {
         return TokenType.PUNCT.of(scanner.consumeOneOf(PUNCT_LIST));
    }

    /**
     * @see SelLexer#consumeKeywordOpt()
     */
    private ScannedToken consumeIdentWithDotsOpt() {
        SelScanner.RangedString rangedString = scanner.consumePattern(IDENT_WITH_DOTS);
        if (rangedString.getValue().contains(".")) {
            return TokenType.IDENT_WITH_DOTS.of(rangedString);
        } else {
            return TokenType.IDENT.of(rangedString);
        }
    }

    /**
     * @see SelLexer#consumeKeywordOpt()
     */
    private ScannedToken consumeDurationOpt() {
        return TokenType.DURATION.of(scanner.consumePattern(DURATION_PATTERN));
    }

    /**
     * @see SelLexer#consumeKeywordOpt()
     */
    private ScannedToken consumeDoubleOpt() {
        return TokenType.DOUBLE.of(scanner.consumePattern(DOUBLE_PATTERN));
    }

    /**
     * @see SelLexer#consumeKeywordOpt()
     */
    private ScannedToken consumeStringLiteralOpt() {
        return TokenType.STRING.of(scanner.consumeQuotedString());
    }

    public ScannedToken nextToken() {
        consumeWhitespaceAndComments();

        if (scanner.lookingAtEof()) {
            return TokenType.EOF.of(scanner.consumeRest());
        }

        for (Supplier<ScannedToken> tokenConsumer : tokenParsingOrder) {
            ScannedToken token = tokenConsumer.get();
            if (token.isPresent()) {
                return token;
            }
        }

        return TokenType.UNRECOGNIZED.of(scanner.consumeRest());
    }

    public List<ScannedToken> nextTokens() {
        List<ScannedToken> r = new ArrayList<>();
        while (true) {
            ScannedToken token = nextToken();
            if (token.getTokenType() == TokenType.EOF) {
                return r;
            }
            r.add(token);
        }
    }

    @Override
    public String toString() {
        return "SelLexer{" +
            "scanner=" + scanner +
            '}';
    }
}
