package ru.yandex.solomon.expression;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelScanner {
    private final LineCountingBuffer buffer;
    public static RangedString FAILED = new RangedString("", PositionRange.UNKNOWN);

    public static class RangedString {
        private final String value;
        private final PositionRange pos;

        public String getValue() {
            return value;
        }

        public RangedString(String value, PositionRange pos) {
            this.value = value;
            this.pos = pos;
        }

        public RangedString(String value, Position begin, Position end) {
            this(value, PositionRange.of(begin, end));
        }

        public PositionRange getPos() {
            return pos;
        }
    }

    public SelScanner(String data) {
        this(new LineCountingBuffer(data));
    }

    private SelScanner(LineCountingBuffer buffer) {
        this.buffer = buffer;
    }

    public SelScanner copy() {
        return new SelScanner(buffer.copy());
    }

    private RangedString consumeUnchecked(String s) {
        var begin = buffer.getCurrentPosition();
        buffer.advance(s);
        var end = s.isEmpty() ? buffer.getCurrentPosition() : buffer.getPrevToCurrentPosition();
        return new RangedString(s, begin, end);
    }


    public RangedString consumePattern(Pattern pattern) {
        String data = buffer.getData();
        Matcher matcher = pattern.matcher(data).region(buffer.getOffset(), data.length());
        if (!matcher.lookingAt()) {
            return FAILED;
        }
        return consumeUnchecked(matcher.group());
    }

    public RangedString consumeRest() {
        var begin = buffer.getCurrentPosition();
        String rest = buffer.readRemaining();
        var end = rest.isEmpty() ? buffer.getCurrentPosition() : buffer.getPrevToCurrentPosition();
        return new RangedString(rest, begin, end);
    }

    public RangedString consume(String s) {
        if (!buffer.getData().startsWith(s, buffer.getOffset())) {
            return FAILED;
        }
        return consumeUnchecked(s);
    }

    public RangedString consumeOneOf(String[] options) {
        for (String s : options) {
            RangedString tryConsume = consume(s);
            if (tryConsume != FAILED) {
                return tryConsume;
            }
        }

        return FAILED;
    }

    public void skipWhitespace() {
        while (true) {
            int c = buffer.peekChar();
            if (c == LineCountingBuffer.EOF || !Character.isWhitespace((char)c)) {
                break;
            }
            buffer.advance((char)c);
        }
    }

    public void skipRestOfLine() {
        while (true) {
            int c = buffer.peekChar();
            if (c == LineCountingBuffer.EOF || c == '\n') {
                break;
            }
            buffer.advance((char)c);
        }
    }

    public RangedString consumeQuotedString() {
        int quote = buffer.peekChar();

        if (quote != '\'' && quote != '"') {
            return FAILED;
        }

        var copy = buffer.copy();
        StringBuilder raw = new StringBuilder();

        copy.advance((char)quote);
        raw.append((char)quote);

        StringBuilder sb = new StringBuilder();
        while (true) {
            int c = copy.readChar();
            if (c == LineCountingBuffer.EOF) {
                return FAILED;
            }
            raw.append((char)c);
            if (c == quote) {
                break;
            }
            if (c != '\\') {
                sb.append((char) c);
                continue;
            }
            int peek = copy.peekChar();
            if (peek == quote) { // TODO: peek == '\\' also should be recognized as single \
                sb.append((char) peek);
                copy.advance((char) peek);
                raw.append((char) peek);
            } else {
                sb.append((char) c);
            }
        }

        RangedString rs = consumeUnchecked(raw.toString());
        return new RangedString(sb.toString(), rs.pos);
    }

    public boolean lookingAtEof() {
        return buffer.peekChar() == LineCountingBuffer.EOF;
    }

    @Override
    public String toString() {
        return "SelScanner{" +
            "buffer=" + buffer +
            '}';
    }
}
