package ru.yandex.solomon.expression.analytics;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Joiner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.expression.ExpressionMetrics;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.AstStatement;
import ru.yandex.solomon.expression.compile.CompileContext;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.expression.compile.SelCompiler;
import ru.yandex.solomon.expression.compile.SelStatement;
import ru.yandex.solomon.expression.exceptions.CompilerException;
import ru.yandex.solomon.expression.exceptions.SelException;
import ru.yandex.solomon.expression.expr.AppendUseToSelectors;
import ru.yandex.solomon.expression.expr.ProgramType;
import ru.yandex.solomon.expression.expr.ProgramTypeVisitor;
import ru.yandex.solomon.expression.expr.SelExprLoadExternalizer;
import ru.yandex.solomon.expression.expr.SelExprParam;
import ru.yandex.solomon.expression.expr.SelExprVisitorFoldConstants;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractCompiler {
    private static final Logger logger = LoggerFactory.getLogger(AbstractCompiler.class);

    protected final String source;
    protected DeprOpts deprOpts;
    protected Boolean useNewFormat;

    protected List<String> externalExpressions = Collections.emptyList();
    protected Map<String, String> selectors = Map.of();

    protected SelVersion version;

    AbstractCompiler(SelVersion version, String src) {
        this.version = version;
        source = src;
    }

    public AbstractCompiler useNewFormat(boolean newFormat) {
        this.useNewFormat = newFormat;
        return this;
    }

    public AbstractCompiler withExternalExpression(String externalExpression) {
        this.externalExpressions = Collections.singletonList(externalExpression);
        return this;
    }

    public AbstractCompiler withExternalExpressions(List<String> externalExpressions) {
        this.externalExpressions = externalExpressions;
        return this;
    }

    public AbstractCompiler withSelectors(Map<String, String> selectors) {
        this.selectors = selectors;
        return this;
    }

    public AbstractCompiler withDeprOpts(DeprOpts deprOpts) {
        this.deprOpts = deprOpts;
        return this;
    }

    Program compileStatements(
            List<AstStatement> statements,
            HashMap<String, String> expressionsToVariableNames)
    {
        return compileStatements(statements, expressionsToVariableNames, new HashMap<>());
    }

    Program compileStatements(
            List<AstStatement> statements,
            HashMap<String, String> expressionsToVariableNames,
            HashMap<String, SelType> contextData)
    {
        CompileContext context = new CompileContext(contextData, deprOpts);

        List<SelStatement> compiled = new SelCompiler(version).compileBlock(statements, context);

        ProgramType programType = ProgramTypeVisitor.computeProgramType(compiled);

        List<SelStatement> compiledFolded;

        compiledFolded = SelExprVisitorFoldConstants.foldBlock(version, compiled, context, selectors);
        compiledFolded = AppendUseToSelectors.fillUse(compiledFolded);
        // no use construct beyond this point

        SelExprLoadExternalizer.ExternalizedResult loadExternalize =
                SelExprLoadExternalizer.externalizeLoad(compiledFolded, useNewFormat);

        Map<SelExprParam, Selectors> selectorByParam = loadExternalize.getExternalParamToSelector();
        List<SelStatement> code = loadExternalize.getPatchedLines();

        if (logger.isDebugEnabled()) {
            logger.debug("Rewrite program:\n{}", Joiner.on("\n").join(code));
        }

        return new Program(version, source, code, selectorByParam, expressionsToVariableNames, programType);
    }

    abstract protected Program compileInternal();

    public Program compile() {
        long startNanos = System.nanoTime();
        try {
            return compileInternal();
        } catch (SelException e) {
            throw e;
        } catch (Throwable t) {
            throw new CompilerException(PositionRange.UNKNOWN, t);
        } finally {
            long took = System.nanoTime() - startNanos;
            ExpressionMetrics.I.compiled(version, took);
        }
    }
}
