package ru.yandex.solomon.expression.analytics;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Objects;

import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.doubles.AggregateFunctionType;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.OperationTop;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Maksim Leonov
 */
@ParametersAreNonnullByDefault
public class GraphDataLoadRequest {
    private final Selectors selectors;
    private final SelType type;
    private final Interval interval;
    private final long gridMillis;

    @Nullable
    private final AggregateFunctionType aggregateFunction;

    @Nullable
    private final OperationTop rankFilter;

    private GraphDataLoadRequest(Builder builder) {
        this.selectors = builder.selectors;
        this.interval = builder.interval;
        this.type = builder.type;
        this.aggregateFunction = builder.aggregateFunction;
        this.gridMillis = builder.gridMillis;
        this.rankFilter = builder.rankFilter;
    }

    public Interval getInterval() {
        return interval;
    }

    public Selectors getSelectors() {
        return selectors;
    }

    public SelType getType() {
        return type;
    }

    @Nullable
    public AggregateFunctionType getAggregateFunction() {
        return aggregateFunction;
    }

    public long getGridMillis() {
        return gridMillis;
    }

    @Nullable
    public OperationTop getRankFilter() {
        return rankFilter;
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        GraphDataLoadRequest that = (GraphDataLoadRequest) o;

        if (gridMillis != that.gridMillis) {
            return false;
        }
        if (!selectors.equals(that.selectors)) {
            return false;
        }
        if (!type.equals(that.type)) {
            return false;
        }
        if (!interval.equals(that.interval)) {
            return false;
        }
        if (!Objects.equal(rankFilter, that.rankFilter)) {
            return false;
        }
        return aggregateFunction == that.aggregateFunction;
    }

    @Override
    public int hashCode() {
        int result = selectors.hashCode();
        result = 31 * result + type.hashCode();
        result = 31 * result + interval.hashCode();
        result = 31 * result + (int) (gridMillis ^ (gridMillis >>> 32));
        result = 31 * result + (rankFilter != null ? rankFilter.hashCode() : 0);
        result = 31 * result + (aggregateFunction != null ? aggregateFunction.hashCode() : 0);
        return result;
    }

    @Override
    public String toString() {
        return "GraphDataLoadRequest{" +
            "selectors=" + Selectors.format(selectors) +
            ", type=" + type +
            ", interval=" + interval +
            ((gridMillis == 0) ? "" : (", gridMillis=" + gridMillis)) +
            ((aggregateFunction == null) ? "" : (", aggregateFunction=" + aggregateFunction)) +
            ((rankFilter == null) ? "" : (", rankFilter=" + rankFilter)) +
            '}';
    }


    public static Builder newBuilder(String selectors) {
        return newBuilder(Selectors.parse(selectors));
    }

    public static Builder newBuilder(Selectors selectors) {
        return new Builder(selectors);
    }

    @ParametersAreNonnullByDefault
    public static class Builder {
        private Selectors selectors;
        private Interval interval;
        private AggregateFunctionType aggregateFunction;
        private long gridMillis;
        private SelType type;
        private OperationTop rankFilter;

        public Builder(Selectors selectors) {
            setSelectors(selectors);
        }

        private Builder(GraphDataLoadRequest request) {
            this.selectors = request.selectors;
            this.interval = request.interval;
            this.aggregateFunction = request.aggregateFunction;
            this.gridMillis = request.gridMillis;
            this.type = request.type;
            this.rankFilter = request.rankFilter;
        }

        public Builder setSelectors(Selectors selectors) {
            this.selectors = selectors;
            return this;
        }

        public Builder setInterval(Interval interval) {
            this.interval = interval;
            return this;
        }

        public Builder setAggregateFunction(AggregateFunctionType aggregateFunction) {
            this.aggregateFunction = aggregateFunction;
            return this;
        }

        public Builder setGridMillis(long gridMillis) {
            this.gridMillis = gridMillis;
            return this;
        }

        public Builder setType(SelType type) {
            this.type = type;
            return this;
        }

        public Builder setRankFilter(boolean asc, int limit, Aggregation aggrFunc) {
            this.rankFilter = OperationTop.newBuilder()
                    .setAsc(asc)
                    .setLimit(limit)
                    .setTimeAggregation(aggrFunc)
                    .build();
            return this;
        }

        public GraphDataLoadRequest build() {
            return new GraphDataLoadRequest(this);
        }
    }
}
