package ru.yandex.solomon.expression.analytics;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.expression.ExpressionMetrics;
import ru.yandex.solomon.expression.compile.SelStatement;
import ru.yandex.solomon.expression.exceptions.EvaluationException;
import ru.yandex.solomon.expression.exceptions.SelException;
import ru.yandex.solomon.expression.expr.EvalContextImpl;
import ru.yandex.solomon.expression.expr.ProgramType;
import ru.yandex.solomon.expression.expr.SelExprParam;
import ru.yandex.solomon.expression.expr.SideEffectProcessor;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueString;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class PreparedProgram {
    private static final Logger logger = LoggerFactory.getLogger(PreparedProgram.class);

    private final String originalCode;
    private final Interval interval;
    private final List<SelStatement> code;
    private final LinkedHashMap<SelExprParam, GraphDataLoadRequest> loadRequests;
    private final Map<String, String> expressionsToVariableNames;
    private final Map<String, SelValue> predefinedVariables;
    private final SideEffectProcessor sideEffectProcessor;
    private final ProgramType programType;
    private final SelVersion version;

    PreparedProgram(
            SelVersion version,
            String originalCode,
            Interval interval,
            List<SelStatement> code,
            LinkedHashMap<SelExprParam, GraphDataLoadRequest> loadRequests,
            Map<String, String> expressionsToVariableNames,
            Map<String, SelValue> predefinedVariables,
            SideEffectProcessor sideEffectProcessor,
            ProgramType programType)
    {
        this.originalCode = originalCode;
        this.interval = interval;
        this.code = code;
        this.loadRequests = loadRequests;
        this.expressionsToVariableNames = expressionsToVariableNames;
        this.predefinedVariables = predefinedVariables;
        this.sideEffectProcessor = sideEffectProcessor;
        this.version = version;
        this.programType = programType;
    }

    public String explainError(SelException e) {
        return Program.explainError(originalCode, e);
    }

    public Map<String, SelValue> evaluate(GraphDataLoader loader, Map<String, String> selectors) {
        long startNanos = System.nanoTime();
        try {
            return evaluateInternal(loader, selectors);
        } finally {
            long took = System.nanoTime() - startNanos;
            ExpressionMetrics.I.evaluated(version, took);
        }
    }

    private Map<String, SelValue> evaluateInternal(GraphDataLoader loader, Map<String, String> selectors) {
        EvalContextImpl interpreterContext = new EvalContextImpl(interval, predefinedVariables, loader, sideEffectProcessor, version);

        selectors.forEach((k, v) -> interpreterContext.addVar(k, new SelValueString(v)));

        for (Map.Entry<SelExprParam, GraphDataLoadRequest> entry : loadRequests.entrySet()) {
            SelExprParam variable = entry.getKey();
            SelValue result = interpreterContext.loadGraphData(variable.getRange(), entry.getValue());
            interpreterContext.addVar(variable.getName(), result);
        }

        for (SelStatement statement : code) {
            try {
                boolean terminate = statement.changeEvalContext(interpreterContext);
                if (terminate) {
                    break;
                }
            } catch (SelException e) {
                throw e;
            } catch (Throwable e) {
                logger.error("Exception while evaluation", e);
                throw new EvaluationException(statement.getRange(), e);
            }
        }

        return interpreterContext.getVars();
    }

    @Nonnull
    public String expressionToVarName() {
        return expressionToVarName(originalCode);
    }

    @Nonnull
    public String expressionToVarName(String expression) {
        String result = expressionsToVariableNames.get(expression);
        if (result != null) {
            return result;
        }

        throw new NullPointerException("Not found variable by key '"
                + expression
                + "' available keys: "
                + expressionsToVariableNames.keySet());
    }

    @VisibleForTesting
    public LinkedHashMap<SelExprParam, GraphDataLoadRequest> getAllLoadRequests() {
        return loadRequests;
    }

    public String getSource() {
        return originalCode;
    }

    public Collection<GraphDataLoadRequest> getLoadRequests() {
        return new HashSet<>(loadRequests.values());
    }

    public SideEffectProcessor getSideEffectProcessor() {
        return sideEffectProcessor;
    }

    public ProgramType getProgramType() {
        return programType;
    }

    public SelVersion getVersion() {
        return version;
    }

    @VisibleForTesting
    public Interval getInterval() {
        return interval;
    }

    @VisibleForTesting
    public List<SelStatement> getCode() {
        return code;
    }

    @VisibleForTesting
    public Map<String, SelValue> getPredefinedVariables() {
        return predefinedVariables;
    }

    public String format() {
        StringBuilder result = new StringBuilder("PreparedProgram{\n");
        result.append("loadRequests=\n");
        loadRequests.forEach((key, value) -> result.append(key.getName()).append(" -> ").append(value).append('\n'));
        result.append("predefinedVariables=\n");
        predefinedVariables.entrySet().stream().sorted(Comparator.comparing(Map.Entry::getKey)).forEach(e ->
            result.append(e.getKey()).append(" -> ").append(e.getValue()).append('\n'));
        result.append("code=\n");
        for (int i = 0; i < code.size(); i++) {
            result.append(i).append("  ").append(code.get(i).format()).append("\n");
        }
        result.append("}");
        return result.toString();
    }
}
