package ru.yandex.solomon.expression.analytics;

import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.expression.ExpressionMetrics;
import ru.yandex.solomon.expression.Position;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.expression.compile.SelStatement;
import ru.yandex.solomon.expression.exceptions.PreparingException;
import ru.yandex.solomon.expression.exceptions.SelException;
import ru.yandex.solomon.expression.expr.IntervalLoadVisitor;
import ru.yandex.solomon.expression.expr.IntrinsicsExternalizer;
import ru.yandex.solomon.expression.expr.ProgramType;
import ru.yandex.solomon.expression.expr.SelExprDownSamplingExternalizer;
import ru.yandex.solomon.expression.expr.SelExprParam;
import ru.yandex.solomon.expression.expr.SelExprRankExternalizer;
import ru.yandex.solomon.expression.expr.SelectorTypeVisitor;
import ru.yandex.solomon.expression.expr.func.util.SelFnEvalTimeInterval;
import ru.yandex.solomon.expression.version.SelVersion;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Maksim Leonov
 * @see <a href="https://wiki.yandex-team.ru/users/nohttp/Solomon-Syntax/">wiki</a> for more info
 */
@ParametersAreNonnullByDefault
public class Program {
    private static final Logger logger = LoggerFactory.getLogger(Program.class);

    private final String originalCode;
    private final List<SelStatement> code;
    private final Map<SelExprParam, Selectors> selectorByParam;
    private final HashMap<String, String> expressionsToVariableNames;
    private final ProgramType programType;
    private final SelVersion version;

    Program(
        SelVersion version,
        String originalCode,
        List<SelStatement> code,
        Map<SelExprParam, Selectors> selectorByParam,
        HashMap<String, String> expressionsToVariableNames, ProgramType programType)
    {
        this.originalCode = originalCode;
        this.code = code;
        this.selectorByParam = selectorByParam;
        this.expressionsToVariableNames = expressionsToVariableNames;
        this.programType = programType;
        this.version = version;
    }

    public boolean isEmpty() {
        return code.isEmpty();
    }

    public ProgramType isSimple() {
        return programType;
    }

    public String getOriginalCode() {
        return originalCode;
    }

    private static String makeUnderline(int line, Position begin, Position end, int codeLen) {
        int start = line > begin.getLine() ? 0 : begin.getColumn() - 1;
        int finish = line < end.getLine() ? codeLen : end.getColumn();
        return StringUtils.repeat(' ', start) +
            (line == begin.getLine() ? "^" : "~") +
            StringUtils.repeat('~', finish - start - 1);
    }

    public static String explainError(String originalCode, SelException e) {
        PositionRange range = e.getRange();
        String line1 = "At " + range + " " + e.getStage() + " error: " + e.getErrorMessage();
        var begin = range.getBegin();
        var end = range.getEnd();

        if (range.equals(PositionRange.UNKNOWN)) {
            return line1;
        }

        try {
            List<String> context = ImmutableList.copyOf(Splitter.on('\n').split(originalCode).iterator());
            String sourcePart = IntStream.range(0, end.getLine() - begin.getLine() + 1)
                .mapToObj(i -> {
                    int line = begin.getLine() + i;
                    String codeLine = context.get(line - 1);
                    String underline = makeUnderline(line, begin, end, codeLine.length());
                    return String.format("% 3d | %s\n      %s", line, codeLine, underline);
                })
                .collect(Collectors.joining("\n"));
            return line1 + "\n" + sourcePart;
        } catch (Throwable t) {
            return line1;
        }
    }

    public static AbstractCompiler fromSource(String src) {
        return fromSource(SelVersion.CURRENT, src);
    }

    public static AbstractCompiler fromSource(SelVersion version, String src) {
        return new ProgramCompiler(version, src);
    }

    @VisibleForTesting
    public static AbstractCompiler fromSourceWithReturn(String src, boolean useNewFormat) {
        return new ProgramWithReturnCompiler(SelVersion.CURRENT, src, useNewFormat);
    }

    public static AbstractCompiler fromSourceWithReturn(SelVersion version, String src, boolean useNewFormat) {
        return new ProgramWithReturnCompiler(version, src, useNewFormat);
    }

    public static AbstractCompiler fromSourceWithReturn(SelVersion version, String src, DeprOpts deprOpts, boolean useNewFormat) {
        return new ProgramWithReturnCompiler(version, src, useNewFormat);
    }

    public PreparedProgram prepare(Interval interval) {
        return prepare(PrepareContext.onInterval(interval).build());
    }

    public PreparedProgram prepare(IntrinsicsExternalizer intrinsicsExternalizer) {
        return prepare(PrepareContext
                        .onInterval(intrinsicsExternalizer.getInterval())
                        .withExternalizer(intrinsicsExternalizer).build());
    }

    public PreparedProgram prepare(PrepareContext prepareContext) {
        long startNanos = System.nanoTime();
        try {
            return prepareInternal(prepareContext);
        } catch (SelException e) {
            throw e;
        } catch (Throwable t) {
            logger.error("Exception while preparing", t);
            throw new PreparingException(PositionRange.UNKNOWN, t);
        } finally {
            long took = System.nanoTime() - startNanos;
            ExpressionMetrics.I.prepared(version, took);
        }
    }

    private PreparedProgram prepareInternal(PrepareContext prepareContext) {
        Interval interval = prepareContext.getInterval();
        IntrinsicsExternalizer intrinsicsExternalizer = prepareContext.getExternalizer();

        List<SelStatement> patched = code;
        patched = intrinsicsExternalizer.fillIntrinsics(version, patched);
        HashMap<SelExprParam, Selectors> paramToSelectors = new HashMap<>(intrinsicsExternalizer.getIntrinsicSelectors());
        paramToSelectors.putAll(selectorByParam);

        Map<SelExprParam, GraphDataLoadRequest.Builder> extVarToRequest = new HashMap<>(paramToSelectors.size());
        for (Map.Entry<SelExprParam, Selectors> entry : paramToSelectors.entrySet()) {
            SelExprParam param = entry.getKey();
            var graphDataLoadRequestBuilder = GraphDataLoadRequest.newBuilder(entry.getValue())
                .setType(param.type());
            extVarToRequest.put(param, graphDataLoadRequestBuilder);
        }

        patched = SelFnEvalTimeInterval.patch(interval, patched);
        patched = SelectorTypeVisitor.fillSelectorType(extVarToRequest, patched);
        patched = SelExprDownSamplingExternalizer.fillDownSampling(extVarToRequest, patched);
        patched = SelExprRankExternalizer.fillRank(extVarToRequest, patched);
        patched = IntervalLoadVisitor.fillInterval(version, interval, extVarToRequest, patched);

        LinkedHashMap<SelExprParam, GraphDataLoadRequest> loadRequests = extVarToRequest.entrySet().stream()
            .sorted(Comparator.comparing(entry -> entry.getKey().getName()))
            .collect(Collectors.toMap(Map.Entry::getKey, entry -> entry.getValue().build(),
                    (l, r) -> r, () -> new LinkedHashMap<>(extVarToRequest.size())));

        if (logger.isDebugEnabled()) {
            logger.debug("Prepared program to execute:\n{}\n\nLoadRequests:\n{}",
                Joiner.on("\n").join(patched),
                Joiner.on("\n")
                    .withKeyValueSeparator(" => ")
                    .join(loadRequests)
            );
        }

        return new PreparedProgram(version, getOriginalCode(), interval, patched, loadRequests,
            expressionsToVariableNames, Map.of(), prepareContext.getSideEffectProcessor(), programType);
    }

    public List<Selectors> getProgramSelectors() {
        return ImmutableList.copyOf(selectorByParam.values());
    }

    // TODO: do not use newlines in toString
    @Override
    public String toString() {
        StringBuilder result = new StringBuilder();
        if (!selectorByParam.isEmpty()) {
            var orderedSelectorByParam = selectorByParam.entrySet().stream()
                    .sorted(Comparator.comparing(e -> e.getKey().getName()))
                    .collect(Collectors.toList());
            for (Map.Entry<SelExprParam, Selectors> staticRequest : orderedSelectorByParam) {
                result.append("# prefetch: ")
                    .append("\n")
                    .append("{").append(Selectors.format(staticRequest.getValue())).append("}")
                    .append(" <- ")
                    .append(staticRequest.getKey().getName())
                    .append("(")
                    .append(staticRequest.getKey().type())
                    .append(")\n");
            }
            result.append("\n");
        }
        for (SelStatement stmt : code) {
            result.append(stmt.format()).append("\n");
        }
        return result.toString();
    }
}
