package ru.yandex.solomon.expression.ast.serialization;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;

import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.ast.AstLambda;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AstLambdaMapper implements AstMapper<AstLambda> {
    private static final String PARAMS_KEY = "params";
    private static final String BODY_KEY = "body";

    AstLambdaMapper() {
    }

    private List<String> parseStringArray(JsonNode json) {
        ArrayList<String> items = new ArrayList<>(json.size());
        json.elements().forEachRemaining(e -> items.add(e.textValue()));
        return items;
    }

    @Override
    public AstLambda parse(AstMappingContext ctx, PositionRange range, JsonNode json) {
        return new AstLambda(range, parseStringArray(json.get(PARAMS_KEY)), ctx.parse(json.get(BODY_KEY)));
    }

    @Override
    public void render(AstMappingContext ctx, AstLambda ast, ObjectNode node) {
        node.set(BODY_KEY, ctx.render(ast.body));
        node.putArray(PARAMS_KEY).addAll(ast.paramNames.stream().map(TextNode::new).collect(Collectors.toList()));
    }

    @Override
    public String renderToString(AstMappingContext ctx, AstLambda ast) {
        String params = ast.paramNames.size() == 1 ? ast.paramNames.get(0) :
                ast.paramNames.stream().collect(Collectors.joining(", ", "(", ")"));
        return params + " -> " + ctx.renderToString(ast.body);
    }

    @Override
    public String getType() {
        return "lambda";
    }

    @Override
    public Class<AstLambda> getAstClass() {
        return AstLambda.class;
    }
}
